/* ************************************************************************
*   File: skills.c                                        EmpireMUD 2.0b5 *
*  Usage: code related to skills, including DB and OLC                    *
*                                                                         *
*  EmpireMUD code base by Paul Clarke, (C) 2000-2024                      *
*  All rights reserved.  See license.doc for complete information.        *
*                                                                         *
*  EmpireMUD based upon CircleMUD 3.0, bpl 17, by Jeremy Elson.           *
*  CircleMUD (C) 1993, 94 by the Trustees of the Johns Hopkins University *
*  CircleMUD is based on DikuMUD, Copyright (C) 1990, 1991.               *
************************************************************************ */

#include "conf.h"
#include "sysdep.h"

#include "structs.h"
#include "utils.h"
#include "comm.h"
#include "interpreter.h"
#include "handler.h"
#include "db.h"
#include "skills.h"
#include "vnums.h"
#include "dg_scripts.h"
#include "olc.h"
#include "constants.h"

/**
* Contents:
*   End Affect When Skill Lost -- core code for shutting off effects
*   Synergy Display Helpers
*   Core Skill Functions
*   Core Skill Commands
*   Helpers
*   Utilities
*   Database
*   OLC Handlers
*   Skill Ability Display
*   Main Displays
*   Edit Modules
*/

// external variables
extern int master_ability_levels[];

// local data
const char *default_skill_name = "Unnamed Skill";
const char *default_skill_abbrev = "???";
const char *default_skill_desc = "New skill";

// local protos
struct skill_ability *find_skill_ability(skill_data *skill, ability_data *abil);
int get_ability_points_available(any_vnum skill, int level);
int get_ability_points_spent(char_data *ch, any_vnum skill);
void get_skill_synergy_display(struct synergy_ability *list, char *save_buffer, char_data *info_ch);
bool green_skill_deadend(char_data *ch, any_vnum skill);
int sort_skill_abilities(struct skill_ability *a, struct skill_ability *b);
int sort_synergies(struct synergy_ability *a, struct synergy_ability *b);


 //////////////////////////////////////////////////////////////////////////////
//// END AFFECT WHEN SKILL LOST //////////////////////////////////////////////

/**
* Code that must run when skills are sold.
*
* @param char_data *ch
* @param ability_data *abil The ability to sell
*/
void check_skill_sell(char_data *ch, ability_data *abil) {
	struct ability_data_list *adl;
	char_data *mob, *next_mob;
	obj_data *obj;
	bool need_affect_total = FALSE;
	int pos;
	
	// empire_data *emp = GET_LOYALTY(ch);
	
	// un-morph
	if (IS_MORPHED(ch) && MORPH_ABILITY(GET_MORPH(ch)) == ABIL_VNUM(abil)) {
		finish_morphing(ch, NULL);
		need_affect_total = TRUE;
	}
	
	// generic affect abilities
	if (affected_by_spell_from_caster(ch, ABIL_AFFECT_VNUM(abil), ch)) {
		affect_from_char_by_caster(ch, ABIL_AFFECT_VNUM(abil), ch, TRUE);
		need_affect_total = TRUE;
	}
	
	// remove any passives
	if (IS_SET(ABIL_TYPES(abil), ABILT_PASSIVE_BUFF)) {
		remove_passive_buff_by_ability(ch, ABIL_VNUM(abil));
		need_affect_total = TRUE;
	}
	
	// remove readied weapons and companions
	LL_FOREACH(ABIL_DATA(abil), adl) {
		switch (adl->type) {
			case ADL_READY_WEAPON: {
				// is the player using the item in any slot
				for (pos = 0; pos < NUM_WEARS; ++pos) {
					if ((obj = GET_EQ(ch, pos)) && GET_OBJ_VNUM(obj) == adl->vnum) {
						act("You stop using $p.", FALSE, ch, obj, NULL, TO_CHAR);
						unequip_char_to_inventory(ch, pos);
						determine_gear_level(ch);
						need_affect_total = TRUE;
					}
				}
				break;
			}
			case ADL_SUMMON_MOB: {
				despawn_charmies(ch, adl->vnum);
				break;
			}
		}
	}
	
	// player tech losses
	if (IS_SET(ABIL_TYPES(abil), ABILT_PLAYER_TECH)) {
		LL_FOREACH(ABIL_DATA(abil), adl) {
			if (adl->type != ADL_PLAYER_TECH) {
				continue;	// wrong type
			}
			if (has_player_tech(ch, adl->vnum)) {
				continue;	// player is still getting this tech from somewhere else
			}
			
			// PTECH_x: what to do when losing the tech
			switch (adl->vnum) {
				case PTECH_ARMOR_HEAVY: {
					remove_armor_by_type(ch, ARMOR_HEAVY);
					need_affect_total = TRUE;
					break;
				}
				case PTECH_ARMOR_LIGHT: {
					remove_armor_by_type(ch, ARMOR_LIGHT);
					need_affect_total = TRUE;
					break;
				}
				case PTECH_ARMOR_MAGE: {
					remove_armor_by_type(ch, ARMOR_MAGE);
					need_affect_total = TRUE;
					break;
				}
				case PTECH_ARMOR_MEDIUM: {
					remove_armor_by_type(ch, ARMOR_MEDIUM);
					need_affect_total = TRUE;
					break;
				}
				case PTECH_BLOCK: {
					if ((obj = GET_EQ(ch, WEAR_HOLD)) && IS_SHIELD(obj)) {
						act("You stop using $p.", FALSE, ch, GET_EQ(ch, WEAR_HOLD), NULL, TO_CHAR);
						unequip_char_to_inventory(ch, WEAR_HOLD);
						determine_gear_level(ch);
						need_affect_total = TRUE;
					}
					break;
				}
				case PTECH_FISH_COMMAND: {
					if (GET_ACTION(ch) == ACT_FISHING) {
						cancel_action(ch);
					}
					break;
				}
				case PTECH_NAVIGATION: {
					// avoid spinning the map when they lose navigation
					GET_CONFUSED_DIR(ch) = NORTH;
					break;
				}
				case PTECH_RANGED_COMBAT: {
					if (GET_EQ(ch, WEAR_RANGED) && IS_MISSILE_WEAPON(GET_EQ(ch, WEAR_RANGED))) {
						act("You stop using $p.", FALSE, ch, GET_EQ(ch, WEAR_RANGED), NULL, TO_CHAR);
						unequip_char_to_inventory(ch, WEAR_RANGED);
						determine_gear_level(ch);
						need_affect_total = TRUE;
					}
					break;
				}
				case PTECH_RIDING: {
					if (IS_RIDING(ch)) {
						msg_to_char(ch, "You climb down from your mount.\r\n");
						perform_dismount(ch);
						need_affect_total = TRUE;
					}
					break;
				}
				case PTECH_USE_HONED_GEAR: {
					remove_honed_gear(ch);
					need_affect_total = TRUE;
					break;
				}
			}
		}
	}
	
	switch (ABIL_VNUM(abil)) {
		case ABIL_BOOST: {
			affect_from_char(ch, ATYPE_BOOST, TRUE);
			need_affect_total = TRUE;
			break;
		}
		case ABIL_DISGUISE: {
			if (IS_DISGUISED(ch)) {
				undisguise(ch);
				need_affect_total = TRUE;
			}
			break;
		}
		case ABIL_EARTHMELD: {
			if (affected_by_spell(ch, ATYPE_EARTHMELD)) {
				un_earthmeld(ch);
				need_affect_total = TRUE;
			}
			break;
		}
		case ABIL_MIRRORIMAGE: {
			DL_FOREACH_SAFE(character_list, mob, next_mob) {
				if (GET_LEADER(mob) == ch && IS_NPC(mob) && GET_MOB_VNUM(mob) == MIRROR_IMAGE_MOB) {
					act("$n vanishes.", TRUE, mob, NULL, NULL, TO_ROOM);
					extract_char(mob);
				}
			}
			break;
		}
	}
	
	determine_gear_level(ch);
	if (need_affect_total) {
		affect_total(ch);
	}
}


 //////////////////////////////////////////////////////////////////////////////
//// SYNERGY DISPLAY HELPERS /////////////////////////////////////////////////

// helper types and functions for creating a clean list of synergy abilities

struct synergy_display_ability {
	any_vnum vnum;	// which abil
	struct synergy_display_ability *next;
};


struct synergy_display_type {
	int role;
	any_vnum skill[2];	// which skills in this display
	any_vnum level[2];	// which level (corresponding to skills)
	struct synergy_display_ability *abils;
	
	struct synergy_display_type *next;
};


/**
* Synergy consolidation for display: Add a synergy ability to a list, only if
* it is not already in the list.
*
* @param struct synergy_display_type **list A pointer to the list to add to.
* @param int role Which role this is fore.
* @param any_vnum skill_a First skill vnum in the pair.
* @param int level_a Which level for skill_a.
* @param any_vnum skill_b Second skill vnum in the pair.
* @param int level_b Which level for skill_b.
* @param any_vnum abil Which ability.
*/
void add_synergy_display_ability(struct synergy_display_type **list, int role, any_vnum skill_a, int level_a, any_vnum skill_b, int level_b, any_vnum abil) {
	struct synergy_display_type *sdt, *iter;
	struct synergy_display_ability *sda;
	
	sdt = NULL;
	
	LL_FOREACH(*list, iter) {
		if (iter->role != role) {
			continue;
		}
		
		if (iter->skill[0] == skill_a && iter->level[0] == level_a && iter->skill[1] == skill_b && iter->level[1] == level_b) {
			sdt = iter;
		}
		else if (iter->skill[1] == skill_a && iter->level[1] == level_a && iter->skill[0] == skill_b && iter->level[0] == level_b) {
			sdt = iter;
		}
		
		if (sdt) {
			break;	// found!
		}
	}
	
	if (!sdt) {	// add skill pair if needed if needed
		CREATE(sdt, struct synergy_display_type, 1);
		sdt->role = role;
		sdt->skill[0] = skill_a;
		sdt->level[0] = level_a;
		sdt->skill[1] = skill_b;
		sdt->level[1] = level_b;
		LL_PREPEND(*list, sdt);
	}
	
	// find the ability
	LL_SEARCH_SCALAR(sdt->abils, sda, vnum, abil);
	if (!sda) {	// add if necessary
		CREATE(sda, struct synergy_display_ability, 1);
		sda->vnum = abil;
		LL_PREPEND(sdt->abils, sda);
	}
}


/**
* Frees all the memory from a synergy display list.
*
* @param struct synergy_display_type *list The list to free.
*/
void free_synergy_display_list(struct synergy_display_type *list) {
	struct synergy_display_ability *sda, *next_sda;
	struct synergy_display_type *iter, *next_iter;
	
	LL_FOREACH_SAFE(list, iter, next_iter) {
		LL_FOREACH_SAFE(iter->abils, sda, next_sda) {
			free(sda);
		}
		free(iter);
	}
}


// Simple sorter for the synergy display
int sort_sdt_list(struct synergy_display_type *a, struct synergy_display_type *b) {
	if (a->role != b->role) {
		return a->role - b->role;
	}
	if (a->skill[0] != b->skill[0]) {
		return a->skill[0] - b->skill[0];
	}
	if (a->skill[1] != b->skill[1]) {
		return a->skill[1] - b->skill[1];
	}
	if (a->level[0] != b->level[0]) {
		return b->level[0] - a->level[0];
	}
	if (a->level[1] != b->level[1]) {
		return b->level[1] - a->level[1];
	}
	
	return 0;	// somehow identical?
}

// Simple sorter for the synergy display abilities
int sort_sda_list(struct synergy_display_ability *a, struct synergy_display_ability *b) {
	char buf1[256], buf2[256];
	strcpy(buf1, get_ability_name_by_vnum(a->vnum));
	strcpy(buf2, get_ability_name_by_vnum(b->vnum));
	return strcmp(buf1, buf2);
}


 //////////////////////////////////////////////////////////////////////////////
//// CORE SKILL FUNCTIONS ////////////////////////////////////////////////////

// for sorting the skill display
struct skill_display_t {
	int level;
	char *name_ptr;	// DO NOT FREE -- only points to the name
	char *string;	// must free this, though
	
	struct skill_display_t *prev, *next;	// doubly-linked list
};


/**
* @param struct skill_display_t *list Frees this list of skill display lines.
*/
void free_skill_display_t(struct skill_display_t *list) {
	struct skill_display_t *iter, *next;
	
	DL_FOREACH_SAFE(list, iter, next) {
		if (iter->string) {
			free(iter->string);
		}
		DL_DELETE(list, iter);
		free(iter);
	}
}


// level sorter for skill displays
int sort_skill_display_by_level(struct skill_display_t *a, struct skill_display_t *b) {
	if (a->level != b->level) {
		return a->level - b->level;
	}
	else {
		// same level
		return strcmp(NULLSAFE(a->name_ptr), NULLSAFE(b->name_ptr));
	}
}


// alpha sorter for skill displays
int sort_skill_display_by_name(struct skill_display_t *a, struct skill_display_t *b) {
	return strcmp(NULLSAFE(a->name_ptr), NULLSAFE(b->name_ptr));
}


/**
* Determines how to color an ability based on its level.
*
* @param char_data *ch The player with the ability
* @param ability_data *abil The ability to color;
*/
char *ability_color(char_data *ch, ability_data *abil) {
	bool can_buy, has_bought, has_maxed;
	struct skill_ability *skab;
	
	if (!abil) {
		return "\tr";
	}
	
	has_bought = has_ability(ch, ABIL_VNUM(abil));
	can_buy = (ABIL_ASSIGNED_SKILL(abil) && get_skill_level(ch, SKILL_VNUM(ABIL_ASSIGNED_SKILL(abil))) >= ABIL_SKILL_LEVEL(abil)) && (skab = find_skill_ability(ABIL_ASSIGNED_SKILL(abil), abil)) && (skab->prerequisite == NO_ABIL || has_ability(ch, skab->prerequisite));
	has_maxed = has_bought && (levels_gained_from_ability(ch, abil) >= GAINS_PER_ABILITY || (!ABIL_ASSIGNED_SKILL(abil) || IS_ANY_SKILL_CAP(ch, SKILL_VNUM(ABIL_ASSIGNED_SKILL(abil))) || !can_gain_skill_from(ch, abil)));
	
	if (has_bought && has_maxed) {
		return "\tg";
	}
	if (has_bought) {
		return "\ty";
	}
	if (can_buy) {
		return "\tc";
	}
	return "\tr";
}


/**
* Gives an ability to a player on any of their skill sets.
*
* @param char_data *ch The player to check.
* @param ability_data *abil Any valid ability.
* @param int skill_set Which skill set number (0..NUM_SKILL_SETS-1).
* @param bool reset_levels If TRUE, wipes out the number of levels gained from the ability.
*/
void add_ability_by_set(char_data *ch, ability_data *abil, int skill_set, bool reset_levels) {
	struct player_ability_data *data = get_ability_data(ch, ABIL_VNUM(abil), TRUE);
	
	if (skill_set < 0 || skill_set >= NUM_SKILL_SETS) {
		log("SYSERR: Attempting to give ability '%s' to player '%s' on invalid skill set '%d'", GET_NAME(ch), ABIL_NAME(abil), skill_set);
		return;
	}
	
	if (data) {
		data->purchased[skill_set] = TRUE;
		if (reset_levels) {
			data->levels_gained = 0;
		}
		
		if (IS_SET(ABIL_TYPES(abil), ABILT_PLAYER_TECH) && skill_set == GET_CURRENT_SKILL_SET(ch)) {
			apply_ability_techs_to_player(ch, abil);
		}
		qt_change_ability(ch, ABIL_VNUM(abil));
	
		// attach gain hooks
		add_ability_gain_hook(ch, abil);
		
		if (skill_set == GET_CURRENT_SKILL_SET(ch) && IS_SET(ABIL_TYPES(abil), ABILT_PASSIVE_BUFF)) {
			apply_one_passive_buff(ch, abil);
		}
	}
}


/**
* Gives an ability to a player on their current skill set.
*
* @param char_data *ch The player to check.
* @param ability_data *abil Any valid ability.
* @param bool reset_levels If TRUE, wipes out the number of levels gained from the ability.
*/
void add_ability(char_data *ch, ability_data *abil, bool reset_levels) {
	add_ability_by_set(ch, abil, GET_CURRENT_SKILL_SET(ch), reset_levels);
}


/** 
* @param char_data *ch The player
* @param ability_data *abil The ability
* @return TRUE if ch can still gain skill from ability
*/
bool can_gain_skill_from(char_data *ch, ability_data *abil) {
	// must have the ability and not gained too many from it
	if (ABIL_ASSIGNED_SKILL(abil) && get_skill_level(ch, SKILL_VNUM(ABIL_ASSIGNED_SKILL(abil))) < SKILL_MAX_LEVEL(ABIL_ASSIGNED_SKILL(abil)) && has_ability(ch, ABIL_VNUM(abil)) && levels_gained_from_ability(ch, abil) < GAINS_PER_ABILITY) {
		// these limit abilities purchased under each cap to players who are still under that cap
		if (ABIL_SKILL_LEVEL(abil) >= BASIC_SKILL_CAP || get_skill_level(ch, SKILL_VNUM(ABIL_ASSIGNED_SKILL(abil))) < BASIC_SKILL_CAP) {
			if (ABIL_SKILL_LEVEL(abil) >= SPECIALTY_SKILL_CAP || get_skill_level(ch, SKILL_VNUM(ABIL_ASSIGNED_SKILL(abil))) < SPECIALTY_SKILL_CAP) {
				if (ABIL_SKILL_LEVEL(abil) >= MAX_SKILL_CAP || get_skill_level(ch, SKILL_VNUM(ABIL_ASSIGNED_SKILL(abil))) < MAX_SKILL_CAP) {
					return TRUE;
				}
			}
		}
	}
	
	return FALSE;
}


/**
* General test function for abilities. It allows mobs who are not charmed to
* use any ability. Otherwise, it supports an energy pool cost and/or cooldowns,
* as well as checking that the player has the ability.
*
* This function will also provide its other functionality for non-abilities if
* you pass NO_ABIL or a negative number to the ability parameter.
*
* @param char_data *ch The player or NPC.
* @param any_vnum ability Any ABIL_ const or vnum (optional, may be NO_ABIL or a negative number to skip ability checks).
* @param int cost_pool HEALTH, MANA, MOVE, BLOOD (NOTHING if no charge).
* @param int cost_amount Mana (or whatever) amount required, if any.
* @param int cooldown_type Any COOLDOWN_ const, or NOTHING for no cooldown check.
* @return bool TRUE if ch can use ability; FALSE if not.
*/
bool can_use_ability(char_data *ch, any_vnum ability, int cost_pool, int cost_amount, int cooldown_type) {
	ability_data *abil = find_ability_by_vnum(ability);
	char buf[MAX_STRING_LENGTH];
	int time, needs_cost;
	
	// purchase check first, or the rest don't make sense.
	if (!IS_NPC(ch) && ability != NO_ABIL && ability >= 0 && !has_ability(ch, ability)) {
		msg_to_char(ch, "You have not purchased the %s ability.\r\n", get_ability_name_by_vnum(ability));
		return FALSE;
	}
	
	// this actually blocks npcs, too, so it's higher than other checks
	if (cost_pool == BLOOD && cost_amount > 0 && !CAN_SPEND_BLOOD(ch)) {
		msg_to_char(ch, "Your blood is inert, you can't do that!\r\n");
		return FALSE;
	}
	
	// special rules for npcs
	if (IS_NPC(ch)) {
		return (AFF_FLAGGED(ch, AFF_CHARM) ? FALSE : TRUE);
	}
	
	// special rule: require that blood or health costs not reduce player below 1
	needs_cost = cost_amount + ((cost_pool == HEALTH || cost_pool == BLOOD) ? 1 : 0);
	
	// more player checks
	if (cost_pool >= 0 && cost_pool < NUM_POOLS && cost_amount > 0 && GET_CURRENT_POOL(ch, cost_pool) < needs_cost) {
		msg_to_char(ch, "You need %d %s point%s to do that.\r\n", cost_amount, pool_types[cost_pool], PLURAL(cost_amount));
		return FALSE;
	}
	if (abil && ABIL_REQUIRES_TOOL(abil) && !has_all_tools(ch, ABIL_REQUIRES_TOOL(abil))) {
		prettier_sprintbit(ABIL_REQUIRES_TOOL(abil), tool_flags, buf);
		if (count_bits(ABIL_REQUIRES_TOOL(abil)) > 1) {
			msg_to_char(ch, "You need tools to do that: %s\r\n", buf);
		}
		else {
			msg_to_char(ch, "You need %s %s to do that.\r\n", AN(buf), buf);
		}
		return FALSE;
	}

	if (cooldown_type != NOTHING && (time = get_cooldown_time(ch, cooldown_type)) > 0) {
		if (time > 60) {
			msg_to_char(ch, "Your %s cooldown still has %s left.\r\n", get_generic_name_by_vnum(cooldown_type), colon_time(time, FALSE, NULL));
		}
		else {
			msg_to_char(ch, "Your %s cooldown still has %d second%s.\r\n", get_generic_name_by_vnum(cooldown_type), time, (time != 1 ? "s" : ""));
		}
		return FALSE;
	}

	return TRUE;
}


/**
* Safely charges a player the cost for using an ability (won't go below 0) and
* applies a cooldown, if applicable. This works on both players and NPCs, but
* only applies the cooldown to players.
*
* @param char_data *ch The player or NPC.
* @param int cost_pool HEALTH, MANA, MOVE, BLOOD (NOTHING if no charge).
* @param int cost_amount Mana (or whatever) amount required, if any.
* @param int cooldown_type Any COOLDOWN_ const to apply (NOTHING for none).
* @param int cooldown_time Cooldown duration, if any.
* @param int wait_type Any WAIT_ const or WAIT_NONE for no command lag.
*/
void charge_ability_cost(char_data *ch, int cost_pool, int cost_amount, int cooldown_type, int cooldown_time, int wait_type) {
	if (cost_pool >= 0 && cost_pool < NUM_POOLS && cost_amount > 0) {
		set_current_pool(ch, cost_pool, GET_CURRENT_POOL(ch, cost_pool) - cost_amount);
		if (GET_CURRENT_POOL(ch, cost_pool) < 0) {
			set_current_pool(ch, cost_pool, 0);
		}
	}
	
	// only npcs get cooldowns here
	if (cooldown_time > 0 && !IS_NPC(ch)) {
		add_cooldown(ch, cooldown_type, cooldown_time);
	}
	
	command_lag(ch, wait_type);
}


/**
* Checks one skill for a player, and removes and abilities that are above the
* players range. If the player is overspent on points after that, the whole
* ability is cleared.
*
* @param char_data *ch the player
* @param any_vnum skill which skill, or NO_SKILL for all
*/
void check_ability_levels(char_data *ch, any_vnum skill) {
	struct player_ability_data *abil, *next_abil;
	bool all = (skill == NO_SKILL);
	ability_data *abd;
	int iter;
	
	if (IS_NPC(ch)) {	// somehow
		return;
	}
	
	HASH_ITER(hh, GET_ABILITY_HASH(ch), abil, next_abil) {
		abd = abil->ptr;
		
		// is assigned to some skill
		if (!ABIL_IS_PURCHASE(abd) || !ABIL_ASSIGNED_SKILL(abd)) {
			continue;
		}
		// matches requested skill
		if (!all && SKILL_VNUM(ABIL_ASSIGNED_SKILL(abd)) != skill) {
			continue;
		}
		
		if (get_skill_level(ch, SKILL_VNUM(ABIL_ASSIGNED_SKILL(abd))) < ABIL_SKILL_LEVEL(abd)) {
			// whoops too low
			for (iter = 0; iter < NUM_SKILL_SETS; ++iter) {
				if (abil->purchased[iter]) {
					remove_ability_by_set(ch, abil->ptr, iter, FALSE);
					if (iter == GET_CURRENT_SKILL_SET(ch)) {
						check_skill_sell(REAL_CHAR(ch), abil->ptr);
					}
				}
			}
		}
	}
	
	// check if they have too many points spent now (e.g. got early points)
	if (get_ability_points_available(skill, get_skill_level(ch, skill)) < get_ability_points_spent(ch, skill)) {
		clear_char_abilities(ch, skill);
	}
	
	queue_delayed_update(ch, CDU_SAVE);
}


/**
* Checks if a player is allowed to gain any skill points in a given skill.
*
* @param char_data *ch The player.
* @param any_vnum skill_vnum The skill to check.
* @return bool TRUE if the player could gain; FALSE if not.
*/
bool check_can_gain_skill(char_data *ch, any_vnum skill_vnum) {
	struct player_skill_data *skdata = get_skill_data(ch, skill_vnum, TRUE);
	return (skdata && !skdata->noskill && !IS_ANY_SKILL_CAP(ch, skill_vnum) && (skdata->level > 0 || CAN_GAIN_NEW_SKILLS(ch)));
}


/**
* removes all abilities for a player in a given skill on their CURRENT skill set
*
* @param char_data *ch the player
* @param any_vnum skill which skill, or NO_SKILL for all
*/
void clear_char_abilities(char_data *ch, any_vnum skill) {
	struct player_ability_data *abil, *next_abil;
	bool all = (skill == NO_SKILL);
	ability_data *abd;
	
	if (!IS_NPC(ch)) {
		HASH_ITER(hh, GET_ABILITY_HASH(ch), abil, next_abil) {
			abd = abil->ptr;
			if (all || (ABIL_ASSIGNED_SKILL(abd) && SKILL_VNUM(ABIL_ASSIGNED_SKILL(abd)) == skill)) {
				if (abil->purchased[GET_CURRENT_SKILL_SET(ch)] && ABIL_SKILL_LEVEL(abd) > 0) {
					remove_ability(ch, abil->ptr, FALSE);
					check_skill_sell(REAL_CHAR(ch), abil->ptr);
				}
			}
		}
		queue_delayed_update(ch, CDU_SAVE);
	}
}


/**
* Currently, you get 15 bonus exp per day.
*
* @param char_data *ch
* @return int the number of points ch can earn per day
*/
int compute_bonus_exp_per_day(char_data *ch) {
	int perdiem = 0;
	
	if (!IS_NPC(ch)) {
		perdiem = config_get_int("num_daily_skill_points");
	}
	
	if (HAS_BONUS_TRAIT(ch, BONUS_EXTRA_DAILY_SKILLS)) {
		perdiem += config_get_int("num_bonus_trait_daily_skills");
	}
	
	if (GET_LOYALTY(ch) && EMPIRE_HAS_TECH(GET_LOYALTY(ch), TECH_BONUS_EXPERIENCE)) {
		perdiem += 5;
	}
	
	return perdiem;
}


/**
* Basic difficulty roll.
*
* @param int level Skill level 0-100.
* @param int difficulty Any DIFF_ const.
* @return bool TRUE for success, FALSE for failure.
*/
bool difficulty_check(int level, int difficulty) {
	int chance;
	
	// always succeeds
	if (difficulty == DIFF_TRIVIAL) {
		return TRUE;
	}
	
	// cap incoming level at 100
	level = MIN(level, MAX_SKILL_CAP);
	
	// apply modifier to level
	chance = (int) (skill_check_difficulty_modifier[difficulty] * (double) level);
	
	return (number(1, 100) <= chance);
}


/**
* Gives a skillup for anybody in the empire with the ability.
*
* @param empire_data *emp the empire to skillup
* @param any_vnum ability Which ABIL_ to gain
* @param double amount The amount of experience to gain
*/
void empire_skillup(empire_data *emp, any_vnum ability, double amount) {
	descriptor_data *d;
	char_data *ch;
	
	for (d = descriptor_list; d; d = d->next) {
		if (STATE(d) == CON_PLAYING && (ch = d->character)) {
			if (GET_LOYALTY(ch) == emp) {
				gain_ability_exp(ch, ability, amount);
				run_ability_hooks(ch, AHOOK_ABILITY, ability, 0, NULL, NULL, NULL, NULL, NOBITS);
			}
		}
	}
}


/**
* Gives a skillup for anybody in the empire with a player tech.
*
* @param empire_data *emp the empire to skillup
* @param int tech Which PTECH_ to gain on
* @param double amount The amount of experience to gain
*/
void empire_player_tech_skillup(empire_data *emp, int tech, double amount) {
	descriptor_data *d;
	char_data *ch;
	
	for (d = descriptor_list; d; d = d->next) {
		if (STATE(d) == CON_PLAYING && (ch = d->character)) {
			if (GET_LOYALTY(ch) == emp) {
				gain_player_tech_exp(ch, tech, amount);
				run_ability_hooks_by_player_tech(ch, tech, NULL, NULL, NULL, NULL);
			}
		}
	}
}


/**
* Determines which skill or ability is referred to by either arg2 or whole_arg,
* depending upon whether or not the first word was "info" (which is dropped).
*
* This prefers, in order: exact skill, abbreviated skill, exact ability,
* abbreviated ability.
*
* This requires multiple strings which are already available in the command
* which called it.
*
* @param char *arg1 The first word in the argument.
* @param char *rest_arg The rest of the argument.
* @param char *whole_arg The original argument ("arg1 rest_arg").
* @param skill_data **found_skill Will bind any found skill to this pointer.
* @param ability_data **found_abil Will bind any found ability to this pointer.
* @return bool TRUE if it found either a skill or ability; FALSE if not.
*/
static bool find_skill_or_ability_for_command(char *arg1, char *rest_arg, char *whole_arg, skill_data **found_skill, ability_data **found_abil) {
	char *use_arg = ((arg1 && !str_cmp(arg1, "info")) ? rest_arg : whole_arg);
	
	*found_skill = NULL;
	*found_abil = NULL;
	
	// in order of precedence:
	if ((*found_skill = find_skill_by_name_exact(use_arg, FALSE))) {
		return TRUE;
	}
	else if ((*found_skill = find_skill_by_name_exact(use_arg, TRUE))) {
		return TRUE;
	}
	else if ((*found_abil = find_ability_by_name_exact(use_arg, FALSE))) {
		return TRUE;
	}
	else if ((*found_abil = find_ability_by_name_exact(use_arg, TRUE))) {
		return TRUE;
	}
	else {
		return FALSE;
	}
}


/**
* This is the main interface for ability-based skill learning. Ability gain
* caps are checked here. The amount to gain is automatically reduced if the
* player has no daily points available.
*
* @param char_data *ch The player character who will gain.
* @param any_vnum ability The ABIL_ const to gain.
* @param double amount The amount to gain (0-100).
*/
void gain_ability_exp(char_data *ch, any_vnum ability, double amount) {
	ability_data *abil = find_ability_by_vnum(ability);
	skill_data *skill = abil ? ABIL_ASSIGNED_SKILL(abil) : NULL;
	
	if (IS_NPC(ch) || !skill) {
		return;
	}
	
	if (!can_gain_skill_from(ch, abil)) {
		return;
	}
		
	// try gain
	if (skill && gain_skill_exp(ch, SKILL_VNUM(skill), amount, abil)) {
		// increment gains from this
		mark_level_gained_from_ability(ch, abil);
	}
}


/**
* Gains experience based on what's giving a player a certain tech.
*
* @param char_data *ch The player gaining exp.
* @param int tech The PTECH_ type that's triggering.
* @param double amount How much exp to gain.
*/
void gain_player_tech_exp(char_data *ch, int tech, double amount) {
	struct player_tech *iter;
	
	if (IS_NPC(ch)) {
		return;
	}
	
	LL_FOREACH(GET_TECHS(ch), iter) {
		if (iter->id == tech && (!iter->check_solo || check_solo_role(ch))) {
			gain_ability_exp(ch, iter->abil, amount);
			run_ability_hooks(ch, AHOOK_ABILITY, iter->abil, 0, ch, NULL, NULL, NULL, NOBITS);
		}
	}
}


/**
* Mostly-raw skill gain/loss -- slightly more checking than set_skill(). This
* will not pass skill cap boundaries. It will NEVER gain you from 0 either.
*
* @param char_data *ch The character who is to gain/lose the skill.
* @param skill_data *skill The skill to gain/lose.
* @param int amount The number of skill points to gain/lose (gaining will stop at any cap).
* @param ability_data *from_abil Optional: Which ability caused this gain, for reporting purposes. (May be NULL.)
* @param bool Returns TRUE if a skill point was gained or lost.
*/
bool gain_skill(char_data *ch, skill_data *skill, int amount, ability_data *from_abil) {
	bool any = FALSE, pos = (amount > 0);
	char abil_buf[256];
	struct player_skill_data *skdata;
	int points;
	
	if (!ch || IS_NPC(ch) || !skill) {
		return FALSE;
	}
	
	if (!IS_APPROVED(ch) && config_get_bool("skill_gain_approval")) {
		return FALSE;
	}
	
	// inability to gain from 0?
	if (get_skill_level(ch, SKILL_VNUM(skill)) == 0 && !CAN_GAIN_NEW_SKILLS(ch)) {
		return FALSE;
	}
	
	// reasons a character would not gain no matter what
	if (amount > 0 && !noskill_ok(ch, SKILL_VNUM(skill))) {
		return FALSE;
	}
	
	// load skill data now (this is the final check)
	if (!(skdata = get_skill_data(ch, SKILL_VNUM(skill), TRUE))) {
		return FALSE;
	}
	
	while (amount != 0 && (amount > 0 || skdata->level > 0) && (amount < 0 || !IS_ANY_SKILL_CAP(ch, SKILL_VNUM(skill)))) {
		any = TRUE;
		if (pos) {
			set_skill(ch, SKILL_VNUM(skill), skdata->level + 1);
			--amount;
		}
		else {
			set_skill(ch, SKILL_VNUM(skill), skdata->level - 1);
			++amount;
		}
	}
	
	if (any) {
		if (from_abil) {
			// reports 1 gain higher than currently-recorded because it's only incremented after the gain is successful
			safe_snprintf(abil_buf, sizeof(abil_buf), " using %s (%d/%d)", ABIL_NAME(from_abil), levels_gained_from_ability(ch, from_abil) + 1, GAINS_PER_ABILITY);
		}
		else {
			*abil_buf = '\0';
		}
		
		// messaging
		if (pos) {	// positive gain
			// notify if desired
			if (SHOW_STATUS_MESSAGES(ch, SM_SKILL_GAINS)) {
				msg_to_char(ch, "\tyYou improve your %s skill to %d%s.\t0\r\n", SKILL_NAME(skill), skdata->level, abil_buf);
				
				if (!PRF_FLAGGED(ch, PRF_NO_TUTORIALS) && (points = get_ability_points_available_for_char(ch, SKILL_VNUM(skill))) > 0) {
					msg_to_char(ch, "\tyYou have %d ability point%s to spend. Type 'skill %s' to see %s.\t0\r\n", points, (points != 1 ? "s" : ""), SKILL_NAME(skill), (points != 1 ? "them" : "it"));
				}
			}
				
			// did we hit a cap? free reset!
			if (IS_ANY_SKILL_CAP(ch, SKILL_VNUM(skill))) {
				skdata->resets = MIN(skdata->resets + 1, MAX_SKILL_RESETS);
				if (SHOW_STATUS_MESSAGES(ch, SM_SKILL_GAINS)) {
					msg_to_char(ch, "\tyYou have earned a free skill reset in %s. Type 'skill reset %s' to use it.\t0\r\n", SKILL_NAME(skill), SKILL_NAME(skill));
				}
			}
			
			// announce
			if (!IS_IMMORTAL(ch) && skdata->level == SKILL_MAX_LEVEL(skill)) {
				log_to_slash_channel_by_name(PLAYER_LOG_CHANNEL, ch, "%s has reached %s %d!", PERS(ch, ch, TRUE), SKILL_NAME(skill), SKILL_MAX_LEVEL(skill));
			}
		}
		else {	// negative gain
			if (SHOW_STATUS_MESSAGES(ch, SM_SKILL_GAINS)) {
				msg_to_char(ch, "\tyYour %s skill drops to %d%s.\t0\r\n", SKILL_NAME(skill), skdata->level, abil_buf);
			}
		}
		
		// update class and progression
		update_class_and_abilities(ch);
		
		// ensure a role
		if (GET_SKILL_LEVEL(ch) >= MAX_SKILL_CAP && GET_CLASS_ROLE(ch) == ROLE_NONE) {
			auto_assign_role(ch, TRUE);
		}
		
		queue_delayed_update(ch, CDU_PASSIVE_BUFFS | CDU_SAVE | CDU_MSDP_SKILLS);
	}
	
	return any;
}


/**
* This is the main interface for leveling skills. Every time a skill gains
* experience, it has a chance to level up. The chance to level up grows as
* the skill approaches 100. Passing an amount of 100 guarantees a skillup,
* if the character can gain at all.
*
* This function also accounts for daily bonus exp, and automatically
* reduces amount if the character has no daily points left.
*
* @param char_data *ch The player character who will gain.
* @param any_vnum skill_vnum The skill to gain experience in.
* @param double amount The amount to gain (0-100).
* @param ability_data *from_abil Optional: Which ability caused this gain, for reporting purposes. (May be NULL.)
* @return bool TRUE if the character gained a skill point from the exp.
*/
bool gain_skill_exp(char_data *ch, any_vnum skill_vnum, double amount, ability_data *from_abil) {
	struct player_skill_data *skdata;
	skill_data *skill;
	bool gained;
	
	// simply sanitation
	if (amount <= 0 || !ch || !(skdata = get_skill_data(ch, skill_vnum, TRUE)) || !(skill = skdata->ptr)) {
		return FALSE;
	}
	
	// reasons a character would not gain no matter what
	if (skdata->noskill || IS_ANY_SKILL_CAP(ch, skill_vnum)) {
		return FALSE;
	}

	// this allows bonus skillups...
	if (GET_DAILY_BONUS_EXPERIENCE(ch) <= 0) {
		amount /= 25.0;
	}
	
	// gain the exp
	skdata->exp += amount;
	
	// can gain at all?
	if (skdata->exp < config_get_int("min_exp_to_roll_skillup")) {
		return FALSE;
	}
	
	// check for skill gain
	gained = (number(1, 100) <= skdata->exp);
	
	if (gained) {
		GET_DAILY_BONUS_EXPERIENCE(ch) = MAX(0, GET_DAILY_BONUS_EXPERIENCE(ch) - 1);
		gained = gain_skill(ch, skill, 1, from_abil);
		update_MSDP_bonus_exp(ch, UPDATE_SOON);
	}
	
	queue_delayed_update(ch, CDU_MSDP_SKILLS);
	return gained;
}


/**
* Get one ability from a player's hash (or add it if missing). Note that the
* add_if_missing param does not guarantee a non-NULL return if the other
* arguments are bad.
*
* @param char_data *ch The player to get the ability for (NPCs always return NULL).
* @param any_vnum abil_id A valid ability number.
* @param bool add_if_missing If TRUE, will add a ability entry if they player has none.
* @return struct player_ability_data* The ability entry if possible, or NULL.
*/
struct player_ability_data *get_ability_data(char_data *ch, any_vnum abil_id, bool add_if_missing) {
	struct player_ability_data *data;
	ability_data *ptr;
	
	// check bounds
	if (!ch || IS_NPC(ch)) {
		return NULL;
	}
	
	HASH_FIND_INT(GET_ABILITY_HASH(ch), &abil_id, data);
	if (!data && add_if_missing && (ptr = find_ability_by_vnum(abil_id))) {
		CREATE(data, struct player_ability_data, 1);
		data->vnum = abil_id;
		data->ptr = ptr;
		HASH_ADD_INT(GET_ABILITY_HASH(ch), vnum, data);
	}
	
	return data;	// may be NULL
}


/**
* Determines what skill level the player uses an ability at. This is usually
* based on the parent skill of the ability for players, and the scaled level
* for NPCs. In all cases, it returns a range of 0-100.
*
* @param char_data *ch The player or NPC.
* @param int Any ABIL_x const.
* @return int A skill level of 0 to 100.
*/
int get_ability_skill_level(char_data *ch, any_vnum ability) {
	ability_data *abd;
	skill_data *sk;
	
	if (IS_NPC(ch)) {
		return get_approximate_level(ch);
	}
	else {
		// players
		if (ability != NO_ABIL && !has_ability(ch, ability)) {
			return 0;
		}
		else if (ability != NO_ABIL && (abd = find_ability_by_vnum(ability)) && (sk = find_assigned_skill(abd, ch))) {
			return get_skill_level(ch, SKILL_VNUM(sk));
		}
		else {
			return MIN(MAX_SKILL_CAP, GET_COMPUTED_LEVEL(ch));
		}
	}
}


/**
* Gets the computed level of a player, or the scaled level of a mob. If the
* mob has not yet been scaled, it picks the lowest valid level for the mob.
*
* @param char_data *ch The player or NPC.
* @return int The computed level.
*/
int get_approximate_level(char_data *ch) {
	int level = 1;
	
	if (IS_NPC(ch)) {
		if (GET_CURRENT_SCALE_LEVEL(ch) > 0) {
			level = GET_CURRENT_SCALE_LEVEL(ch);
		}
		else {
			if (GET_MAX_SCALE_LEVEL(ch) > 0) {
				level = MIN(GET_MAX_SCALE_LEVEL(ch), level);
			}
			if (GET_MIN_SCALE_LEVEL(ch) > 0) {
				level = MAX(GET_MIN_SCALE_LEVEL(ch), level);
			}
		}
	}
	else {
		// player
		level = GET_COMPUTED_LEVEL(ch);
	}
	
	return level;
}


/**
* @param any_vnum skill Which skill
* @param int level at What level
* @return int How many abilities are available by that level
*/
int get_ability_points_available(any_vnum skill, int level) {
	int iter, count;
	
	count = 0;
	for (iter = 0; master_ability_levels[iter] != -1; ++iter) {
		if (level >= master_ability_levels[iter]) {
			++count;
		}
	}
	
	return count;
}


/**
* @param char_data *ch the person to check
* @param any_vnum skill Which skill to check
* @return int how many ability points ch has available in skill
*/
int get_ability_points_available_for_char(char_data *ch, any_vnum skill) {
	int max = get_ability_points_available(skill, NEXT_CAP_LEVEL(ch, skill));
	int spent = get_ability_points_spent(ch, skill);
	int avail = MAX(0, get_ability_points_available(skill, get_skill_level(ch, skill)) - spent);
	
	// allow early if they're at a deadend
	if (avail == 0 && spent < max && get_skill_level(ch, skill) > 0 && green_skill_deadend(ch, skill)) {
		return 1;
	}
	else {
		return avail;
	}
}


/**
* @param char_data *ch the person to check
* @param any_vnum skill Which skill to check
* @return int total abilities bought in a skill
*/
int get_ability_points_spent(char_data *ch, any_vnum skill) {
	struct skill_ability *skab;
	skill_data *skd;
	int count = 0;
	
	if (!(skd = find_skill_by_vnum(skill))) {
		return count;
	}
	
	LL_FOREACH(SKILL_ABILITIES(skd), skab) {
		if (skab->level <= 0) {
			continue;	// skip level-0 abils
		}
		if (!has_ability(ch, skab->vnum)) {
			continue;	// does not have ability
		}
		if (has_bonus_ability(ch, skab->vnum)) {
			continue;	// don't count bonus abilities as real purchases
		}
		
		// found
		++count;
	}
	
	return count;
}


/**
* Builds the list of abilities for a skill as a doubly-linked list. The order
* of this list is essentially alphabetic but with dependencies shown under
* the parent ability. You can sort these afterwards or just display them in
* order.
*
* @param struct skill_display_t **list Pointer to the list of skills we're building.
* @param char_data *ch The character whose skills to use, and who to send to.
* @param skill_data *skill Which skill to show.
* @param any_vnum prereq Which ability, for dependent abilities (NO_PREREQ for base abils).
* @param int indent How far to indent (goes up as the list goes deeper); use -1 for do-not-indent.
* @param int min_level Optional: Don't show skills below this level (-1 for no minimum).
* @param int max_level Optional: Don't show skills above this level (-1 for no maximum).
* @return string the display
*/
void get_skill_abilities_display(struct skill_display_t **list, char_data *ch, skill_data *skill, any_vnum prereq, int indent, int min_level, int max_level) {
	char out[MAX_STRING_LENGTH], lbuf[MAX_STRING_LENGTH], colorize[16];
	struct skill_ability *skab;
	int ind, max_skill = 0;
	ability_data *abil;
	struct skill_display_t *skdat;
	
	if (!list || !ch || !skill) {
		// oops no work
		return;
	}
	
	LL_FOREACH(SKILL_ABILITIES(skill), skab) {
		if (skab->prerequisite != prereq) {
			continue;
		}
		if (!(abil = find_ability_by_vnum(skab->vnum))) {
			continue;
		}
		
		*out = *lbuf ='\0';
		
		// indent
		if (indent != -1) {
			for (ind = 0; ind < (2 * indent); ++ind) {
				lbuf[ind] = ' ';
			}
			lbuf[2 * indent] = '\0';
		}
		
		if (prereq != NO_PREREQ && indent != -1) {
			strcat(lbuf, "+ ");
		}
		
		if (skab->level < BASIC_SKILL_CAP) {
			max_skill = BASIC_SKILL_CAP;
		}
		else if (skab->level < SPECIALTY_SKILL_CAP) {
			max_skill = SPECIALTY_SKILL_CAP;
		}
		else if (skab->level <= MAX_SKILL_CAP) {
			max_skill = MAX_SKILL_CAP;
		}
		
		// get the proper color for this ability
		strcpy(colorize, ability_color(ch, abil));
		
		if (PRF_FLAGGED(ch, PRF_SCREEN_READER)) {
			sprintf(out + strlen(out), "%s%s%s\t0 - %d-%d, %s", lbuf, colorize, ABIL_NAME(abil), skab->level, max_skill, (has_ability(ch, ABIL_VNUM(abil)) ? (skab->level == 0 ? "free" : "purchased") : "unpurchased"));
		}
		else {	// non-screenreader
			sprintf(out + strlen(out), "%s(%s) %s%s\t0 [%d-%d]", lbuf, (has_ability(ch, ABIL_VNUM(abil)) ? "x" : " "), colorize, ABIL_NAME(abil), skab->level, max_skill);
		}
		
		if (has_ability(ch, ABIL_VNUM(abil))) {
			// this is kind of a hack to quickly make sure you can still use the ability
			if (levels_gained_from_ability(ch, abil) < GAINS_PER_ABILITY && !strcmp(colorize, "\ty")) {
				sprintf(out + strlen(out), "%s %d/%d skill levels gained", (PRF_FLAGGED(ch, PRF_SCREEN_READER) ? "," : ""), levels_gained_from_ability(ch, abil), GAINS_PER_ABILITY);
			}
			else {
				strcat(out, " (max)");
			}
		}
		else {
			// does not have the ability at all
			// we check that the parent skill exists first, since this reports it
			
			if (skab->prerequisite != NO_PREREQ && !has_ability(ch, skab->prerequisite)) {
				sprintf(out + strlen(out), "%s requires %s", (PRF_FLAGGED(ch, PRF_SCREEN_READER) ? "," : ""), get_ability_name_by_vnum(skab->prerequisite));
			}
			else if (get_skill_level(ch, SKILL_VNUM(skill)) < skab->level) {
				sprintf(out + strlen(out), "%s requires %s %d", (PRF_FLAGGED(ch, PRF_SCREEN_READER) ? "," : ""), SKILL_NAME(skill), skab->level);
			}
			else if (!PRF_FLAGGED(ch, PRF_SCREEN_READER)) {
				// (screenreaders see this elsewhere)
				strcat(out, " unpurchased");
			}
		}
		
		strcat(out, "\r\n");
		CREATE(skdat, struct skill_display_t, 1);
		skdat->level = skab->level;
		skdat->name_ptr = ABIL_NAME(abil);
		skdat->string = str_dup(out);
		DL_APPEND(*list, skdat);

		// dependencies
		get_skill_abilities_display(list, ch, skill, skab->vnum, indent == -1 ? indent : (indent+1), min_level, max_level);
	}
}


/**
* Get one skill from a player's hash (or add it if missing). Note that the
* add_if_missing param does not guarantee a non-NULL return if the other
* arguments are bad.
*
* @param char_data *ch The player to get skills for (NPCs always return NULL).
* @param any_vnum vnum A valid skill number.
* @param bool add_if_missing If TRUE, will add a skill entry if they player has none.
* @return struct player_skill_data* The skill entry if possible, or NULL.
*/
struct player_skill_data *get_skill_data(char_data *ch, any_vnum vnum, bool add_if_missing) {
	struct player_skill_data *sk = NULL;
	skill_data *ptr;
	
	// check bounds
	if (!ch || IS_NPC(ch)) {
		return NULL;
	}
	
	HASH_FIND_INT(GET_SKILL_HASH(ch), &vnum, sk);
	if (!sk && add_if_missing && (ptr = find_skill_by_vnum(vnum))) {
		CREATE(sk, struct player_skill_data, 1);
		sk->vnum = vnum;
		sk->ptr = ptr;
		HASH_ADD_INT(GET_SKILL_HASH(ch), vnum, sk);
	}
	
	return sk;	// may be NULL
}


/**
* Gets the display text for how many points you have available.
*
* @param char_data *ch The player.
* @return char* The display text.
*/
char *get_skill_gain_display(char_data *ch) {
	static char out[MAX_STRING_LENGTH];
	
	*out = '\0';
	if (!IS_NPC(ch)) {
		sprintf(out + strlen(out), "You have %d bonus experience point%s available today.\r\n", GET_DAILY_BONUS_EXPERIENCE(ch), PLURAL(GET_DAILY_BONUS_EXPERIENCE(ch)));
		if (!PRF_FLAGGED(ch, PRF_NO_TUTORIALS)) {
			sprintf(out + strlen(out), "Use 'noskill <skill>' to toggle skill gain on and off.\r\n");
		}
	}
	
	return out;
}


/**
* Get the info row for one skill.
*
* @param char_data *ch The person whose numbers to use.
* @param skill_data *skill Which skill.
* @return char* The display.
*/
char *get_skill_row_display(char_data *ch, skill_data *skill) {
	static char out[MAX_STRING_LENGTH];
	struct player_skill_data *skdata;
	char avail[256], experience[256], gain_part[256];
	int points = get_ability_points_available_for_char(ch, SKILL_VNUM(skill));
	
	skdata = get_skill_data(ch, SKILL_VNUM(skill), FALSE);
	
	if (skdata && !skdata->noskill && !IS_ANY_SKILL_CAP(ch, SKILL_VNUM(skill))) {
		sprintf(experience, ", %.1f%% exp", skdata->exp);
	}
	else {
		*experience = '\0';
	}
	
	if (IS_ANY_SKILL_CAP(ch, SKILL_VNUM(skill))) {
		safe_snprintf(gain_part, sizeof(gain_part), "\tymax\t0%s", (skdata && skdata->noskill) ? ", \trnoskill\t0" : "");
	}
	else if (skdata && skdata->noskill) {
		safe_snprintf(gain_part, sizeof(gain_part), "\trnoskill\t0");
	}
	else {
		safe_snprintf(gain_part, sizeof(gain_part), "\tcgaining\t0");
	}
	
	if (points > 0) {
		safe_snprintf(avail, sizeof(avail), ", %d point%s available", points, PLURAL(points));
	}
	else {
		*avail = '\0';
	}
	
	sprintf(out, "[%3d] %s%s\t0 (%s%s%s) - %s\r\n", (skdata ? skdata->level : 0), IS_ANY_SKILL_CAP(ch, SKILL_VNUM(skill)) ? "\tg" : "\ty", SKILL_NAME(skill), gain_part, experience, avail, SKILL_DESC(skill));
	return out;
}


/**
* Determines if a player has hit a deadend with abilities they can level off
* of.
*
* @param char_data *ch The player.
* @param any_vnum skill Any SKILL_ const or vnum.
* @return bool TRUE if the player has found no levelable abilities
*/
bool green_skill_deadend(char_data *ch, any_vnum skill) {
	int avail = get_ability_points_available(skill, get_skill_level(ch, skill)) - get_ability_points_spent(ch, skill);
	skill_data *skdata = find_skill_by_vnum(skill);
	struct skill_ability *skab;
	bool yellow = FALSE;
		
	if (avail <= 0 && skdata) {
		LL_FOREACH(SKILL_ABILITIES(skdata), skab) {
			yellow |= can_gain_skill_from(ch, find_ability_by_vnum(skab->vnum));
		}
	}
	
	// true if we found no yellows
	return !yellow;
}


/**
* Determines if a player has any skill with a given flag. If so, returns the
* player's level in that skill. If more than one skill matches, it returns the
* highest level among them.
*
* @param char_data *ch The player to check.
* @param bitvector_t skill_flag The SKILLF_ to check for (for multiple flags, will return if ANY of them are present).
* @return int If the player has a matching skill, returns the level of that skill. If not, returns 0.
*/
int has_skill_flagged(char_data *ch, bitvector_t skill_flag) {
	struct player_skill_data *plsk, *next_plsk;
	int skill_level = 0;
	
	// npcs lack skills / shortcuts out
	if (!ch || IS_NPC(ch) || !skill_flag) {
		return 0;
	}
	
	HASH_ITER(hh, GET_SKILL_HASH(ch), plsk, next_plsk) {
		if (plsk->ptr && SKILL_FLAGGED(plsk->ptr, skill_flag)) {
			skill_level = MAX(skill_level, plsk->level);
		}
	}
	
	return skill_level;	// or 0 for none
}


/**
* Adds one to the number of levels ch gained from an ability.
*
* @param char_data *ch The player to check.
* @param ability_data *abil Any valid ability.
*/
void mark_level_gained_from_ability(char_data *ch, ability_data *abil) {
	struct player_ability_data *data = get_ability_data(ch, ABIL_VNUM(abil), TRUE);
	if (data) {
		data->levels_gained += 1;
	}
}


/**
* Switches a player's current skill set from one to the other.
*
* @param char_data *ch The player to swap.
*/
void perform_swap_skill_sets(char_data *ch) {
	struct player_ability_data *plab, *next_plab;
	int cur_set, old_set;
	ability_data *abil;
	
	if (IS_NPC(ch)) { // somehow...
		return;
	}
	
	// note: if you ever raise NUM_SKILL_SETS, this is going to need an update
	old_set = GET_CURRENT_SKILL_SET(ch);
	cur_set = (old_set == 1) ? 0 : 1;
	
	// update skill set
	GET_CURRENT_SKILL_SET(ch) = cur_set;
	
	// update abilities:
	HASH_ITER(hh, GET_ABILITY_HASH(ch), plab, next_plab) {
		abil = plab->ptr;
		
		if (ABIL_IS_PURCHASE(abil)) {	// skill ability
			if (plab->purchased[cur_set] && !plab->purchased[old_set]) {
				// added
				apply_ability_techs_to_player(ch, abil);
				qt_change_ability(ch, ABIL_VNUM(abil));
			}
			else if (plab->purchased[old_set] && !plab->purchased[cur_set]) {
				// removed
				remove_player_tech(ch, ABIL_VNUM(abil));
				check_skill_sell(ch, abil);
				qt_change_ability(ch, ABIL_VNUM(abil));
			}
		}
		else {	// non-purchased ability: just ensure it matches the old one; updates itself in assign_class_abilites
			plab->purchased[cur_set] = plab->purchased[old_set];
			qt_change_ability(ch, ABIL_VNUM(abil));	// in case
		}
	}
	
	// call this at the end just in case
	assign_class_and_extra_abilities(ch, NULL, NOTHING);
	affect_total(ch);
	
	queue_delayed_update(ch, CDU_PASSIVE_BUFFS);
}


/**
* Takes an ability away from a player, on a given skill set.
*
* @param char_data *ch The player to check.
* @param ability_data *abil Any ability.
* @param int skill_set Which skill set number (0..NUM_SKILL_SETS-1).
* @param bool reset_levels If TRUE, wipes out the number of levels gained from the ability.
*/
void remove_ability_by_set(char_data *ch, ability_data *abil, int skill_set, bool reset_levels) {
	struct player_ability_data *data = get_ability_data(ch, ABIL_VNUM(abil), FALSE);
	
	if (skill_set < 0 || skill_set >= NUM_SKILL_SETS) {
		log("SYSERR: Attempting to remove ability '%s' to player '%s' on invalid skill set '%d'", GET_NAME(ch), ABIL_NAME(abil), skill_set);
		return;
	}
	
	if (data) {
		data->purchased[skill_set] = FALSE;
		
		if (reset_levels) {
			data->levels_gained = 0;
		}
		
		if (IS_SET(ABIL_TYPES(abil), ABILT_PLAYER_TECH) && skill_set == GET_CURRENT_SKILL_SET(ch)) {
			remove_player_tech(ch, ABIL_VNUM(abil));
		}
		
		qt_change_ability(ch, ABIL_VNUM(abil));
	}
}


/**
* Takes an ability away from a player on their CURRENT set.
*
* @param char_data *ch The player to check.
* @param ability_data *abil Any ability.
* @param bool reset_levels If TRUE, wipes out the number of levels gained from the ability.
*/
void remove_ability(char_data *ch, ability_data *abil, bool reset_levels) {
	remove_ability_by_set(ch, abil, GET_CURRENT_SKILL_SET(ch), reset_levels);
}


/**
* Removes all skills with a given flag/flags from a player.
*
* @param char_data *ch The player.
* @param bitvector_t skill_flag The SKILLF_ flag(s) to match. If you specify more than one, they must all be present.
* @return bool TRUE if any skills were removed.
*/
bool remove_skills_by_flag(char_data *ch, bitvector_t skill_flag) {
	struct player_skill_data *plsk, *next_plsk;
	bool any = FALSE;
	
	if (IS_NPC(ch)) {
		return FALSE;	// no skills
	}
	
	HASH_ITER(hh, GET_SKILL_HASH(ch), plsk, next_plsk) {
		if (plsk->ptr && (SKILL_FLAGS(plsk->ptr) & skill_flag) == skill_flag && plsk->level > 0) {
			set_skill(ch, plsk->vnum, 0);
			clear_char_abilities(ch, plsk->vnum);
			any = TRUE;
		}
	}
	
	if (any) {
		update_class_and_abilities(ch);
	}
	
	return any;
}


/**
* When a character loses skill levels, we clear the "levels_gained" tracker
* for abilities that are >= their new level, so they can successfully regain
* those levels later.
*
* @param char_data *ch The player.
* @param any_vnum skill Which skill.
*/
void reset_skill_gain_tracker_on_abilities_above_level(char_data *ch, any_vnum skill) {
	struct player_ability_data *abil, *next_abil;
	ability_data *ability;
	
	if (!IS_NPC(ch)) {
		HASH_ITER(hh, GET_ABILITY_HASH(ch), abil, next_abil) {
			ability = abil->ptr;
			if (ABIL_ASSIGNED_SKILL(ability) && SKILL_VNUM(ABIL_ASSIGNED_SKILL(ability)) == skill && ABIL_SKILL_LEVEL(ability) >= get_skill_level(ch, skill)) {
				abil->levels_gained = 0;
			}
		}
	}
}


// set a skill directly to a level
void set_skill(char_data *ch, any_vnum skill, int level) {
	struct player_skill_data *skdata;
	bool gain = FALSE;
	bool was_vampire = IS_VAMPIRE(ch);
	
	if ((skdata = get_skill_data(ch, skill, TRUE))) {
		gain = (level > skdata->level);
		
		skdata->level = level;
		skdata->exp = 0.0;
		
		if (!gain) {
			reset_skill_gain_tracker_on_abilities_above_level(ch, skill);
		}
		
		qt_change_skill_level(ch, skill);
		
		// ensure they are a vampire
		if (!was_vampire && IS_VAMPIRE(ch)) {
			make_vampire(ch, TRUE, NOTHING);
		}
		
		queue_delayed_update(ch, CDU_MSDP_SKILLS);
	}
}


/**
* @param char_data *ch who to check for
* @param any_vnum ability which ABIL_x
* @param int difficulty any DIFF_ const
* @return bool TRUE for success, FALSE for fail
*/
bool skill_check(char_data *ch, any_vnum ability, int difficulty) {
	int chance = get_ability_skill_level(ch, ability);
	
	// always succeeds
	if (difficulty == DIFF_TRIVIAL) {
		return TRUE;
	}

	// players without the ability have no chance	
	if (!IS_NPC(ch) && !has_ability(ch, ability)) {
		chance = 0;
	}
	
	return difficulty_check(chance, difficulty);
}


/**
* Runs a skill check based on a tech (when you don't know the actual ability).
* Synergy abilities will check solo role status and may not be valid if the
* player is in solo role but not solo.
*
* @param char_data *ch The person doing the skill check.
* @param int tech Which PTECH_ type.
* @param int difficulty Any DIFF_ const.
* @return bool TRUE if passed, FALSE if failed.
*/
bool player_tech_skill_check(char_data *ch, int tech, int difficulty) {
	any_vnum best_abil = NOTHING;
	struct player_tech *iter;
	int lev, best_level = 0;
	
	if (IS_NPC(ch)) {
		return FALSE;
	}
	
	LL_FOREACH(GET_TECHS(ch), iter) {
		if (iter->id != tech) {
			continue;	// wrong tech
		}
		if (iter->check_solo && !check_solo_role(ch)) {
			continue;	// not solo
		}
		
		lev = get_ability_skill_level(ch, iter->abil);
		
		if (lev > best_level) {
			best_level = lev;
			best_abil = iter->abil;
		}
	}
	
	if (best_abil != NOTHING) {
		return skill_check(ch, best_abil, difficulty);
	}
	else {
		return FALSE;	// no abil
	}
}


/**
* Runs a skill check based on a tech (when you don't know the actual ability).
* This uses the ability's own difficulty setting. Synergy abilities will ensure
* that solo-role characters are solo.
*
* @param char_data *ch The person doing the skill check.
* @param int tech Which PTECH_ type.
* @return bool TRUE if passed, FALSE if failed.
*/
bool player_tech_skill_check_by_ability_difficulty(char_data *ch, int tech) {
	ability_data *abil;
	struct player_tech *iter;
	
	if (IS_NPC(ch)) {
		return FALSE;
	}
	
	LL_FOREACH(GET_TECHS(ch), iter) {
		if (iter->id != tech) {
			continue;	// wrong tech
		}
		if (iter->check_solo && !check_solo_role(ch)) {
			continue;	// solo role but not solo
		}
		if (!(abil = ability_proto(iter->abil))) {
			continue;	// no data?
		}
		
		if (skill_check(ch, iter->abil, ABIL_DIFFICULTY(abil))) {
			return TRUE;
		}
	}
	
	return FALSE;	// no abil or no success
}


 //////////////////////////////////////////////////////////////////////////////
//// CORE SKILL COMMANDS /////////////////////////////////////////////////////

ACMD(do_ability) {
	ability_data *abil;
	
	skip_spaces(&argument);
	
	// optional info arg (changes nothing)
	if (!strn_cmp(argument, "info ", 5)) {
		argument += 4;
		skip_spaces(&argument);
	}
	
	if (!*argument) {
		msg_to_char(ch, "Get info for which ability?\r\n");
	}
	else if (!(abil = find_ability_by_name_exact(argument, TRUE))) {
		// no matching abil
		if (find_skill_by_name_exact(argument, TRUE)) {
			msg_to_char(ch, "Invalid ability. Try 'skill %s' instead.\r\n", argument);
		}
		else {
			msg_to_char(ch, "Unknown ability '%s'.\r\n", argument);
		}
	}
	else {
		// ability details
		show_ability_info(ch, abil, NULL, TRUE);
	}
}


ACMD(do_noskill) {
	struct player_skill_data *skdata;
	skill_data *skill;

	skip_spaces(&argument);
	
	if (IS_NPC(ch)) {
		return;
	}
	else if (!*argument) {
		msg_to_char(ch, "Usage: noskill <skill>\r\n");
		msg_to_char(ch, "Type 'skills' to see which skills are gaining and which are not.\r\n");
	}
	else if (!(skill = find_skill_by_name(argument)) || !(skdata = get_skill_data(ch, SKILL_VNUM(skill), TRUE))) {
		msg_to_char(ch, "Unknown skill.\r\n");
	}
	else {
		if (skdata->noskill) {
			skdata->noskill = FALSE;
			msg_to_char(ch, "You will now \tcbe able to gain\t0 %s skill.\r\n", SKILL_NAME(skill));
		}
		else {
			skdata->noskill = TRUE;
			msg_to_char(ch, "You will \trno longer\t0 gain %s skill.\r\n", SKILL_NAME(skill));
		}
		
		qt_change_skill_level(ch, skdata->vnum);
		queue_delayed_update(ch, CDU_MSDP_SKILLS);
	}
}


ACMD(do_skills) {
	char arg[MAX_INPUT_LENGTH], arg2[MAX_INPUT_LENGTH], lbuf[MAX_STRING_LENGTH], *ptr;
	char new_arg[MAX_INPUT_LENGTH], whole_arg[MAX_INPUT_LENGTH];
	struct skill_display_t *skdat_list = NULL, *skdat;
	struct synergy_display_type *sdt_list = NULL, *sdt;
	struct synergy_display_ability *sda;
	struct player_ability_data *plab, *next_plab;
	struct player_skill_data *skdata;
	skill_data *skill, *next_skill, *synergy[2];
	struct synergy_ability *syn;
	struct skill_ability *skab;
	ability_data *abil;
	int points, level, iter;
	bool found, any, line;
	bool sort_alpha = FALSE, sort_level = FALSE, want_min = FALSE, want_max = FALSE, want_all = FALSE, show_all_info = FALSE;
	int min_level = -1, max_level = -1;
	struct player_bonus_ability *bonus_abil, *next_bonus_abil;
	
	// attempt to parse the args first, to get -l [range] or -a
	if (*argument) {
		// will remove any requests/flags and leave the rest of the arg in new_arg
		*new_arg = '\0';
		while (*argument) {
			argument = any_one_word(argument, arg);
			if (want_min && isdigit(*arg)) {
				min_level = atoi(arg);
				want_min = FALSE;
				
				// look for max, too
				want_max = FALSE;
				
				// look for maximum
				for (iter = 0; arg[iter]; ++iter) {
					if (isdigit(arg[iter]) && !want_max) {
						// skipping levels
					}
					else if (isdigit(arg[iter]) && want_max) {
						max_level = atoi(arg + iter);
						want_max = FALSE;
					}
					else if (arg[iter] == '-' || arg[iter] == ':') {
						// level divider
						want_max = TRUE;
					}
					else {
						// found something else?
						msg_to_char(ch, "Usage: skills [name] -l <level range>\r\n");
						return;
					}
				}
				
				// did we find a max? if not, ask for it in the next arg
				if (max_level == -1) {
					want_max = TRUE;
				}
			}
			else if (want_max && isdigit(*arg)) {
				max_level = atoi(arg);
				want_max = FALSE;
			}
			else if (!strn_cmp(arg, "-l", 2)) {
				sort_level = TRUE;
				
				// check for possible min/max level: find start pos
				if (!strn_cmp(arg, "-level", 6)) {
					iter = 6;
				}
				else if (!strn_cmp(arg, "-lev", 4)) {
					iter = 4;
				}
				else {
					iter = 2;
				}
				
				// look for levels here
				if (iter > strlen(arg) || !arg[iter]) {
					// no level -- free to continue, but look for levels in the next arg
					want_min = want_max = TRUE;
				}
				else if (!isdigit(arg[iter])) {
					// unexpected
					msg_to_char(ch, "Usage: skills [name] -l <level range>\r\n");
					return;
				}
				else {
					// found a probably minimum
					min_level = atoi(arg+iter);
					want_min = want_max = FALSE;
					
					// look for maximum
					for (; arg[iter]; ++iter) {
						if (isdigit(arg[iter]) && !want_max) {
							// skipping levels
						}
						else if (isdigit(arg[iter]) && want_max) {
							max_level = atoi(arg + iter);
							want_max = FALSE;
						}
						else if (arg[iter] == '-' || arg[iter] == ':') {
							// level divider
							want_max = TRUE;
						}
						else {
							// found something else?
							msg_to_char(ch, "Usage: skills [name] -l <level range>\r\n");
							return;
						}
					}
					
					// did we find a max? if not, ask for it in the next arg
					if (max_level == -1) {
						want_max = TRUE;
					}
				}
			}
			else if (is_abbrev(arg, "-all")) {
				want_all = TRUE;
			}
			else if (is_abbrev(arg, "-alphabetical")) {
				sort_alpha = TRUE;
			}
			else {
				// apply it to new_arg instead: length OK because it cannot get LONGER than max-input-length when removing
				sprintf(new_arg + strlen(new_arg), "%s%s", (*new_arg ? " " : ""), arg);
			}
		}
		
		// and keep only new_arg
		argument = new_arg;
	}	// end basic argument parsing
	
	argument = trim(argument);
	strcpy(whole_arg, argument);
	half_chop(argument, arg, arg2);
	
	if (IS_NPC(ch)) {
		msg_to_char(ch, "As an NPC, you have no skills.\r\n");
		return;
	}
	
	// mode based on args
	if (!*arg) {
		// no remaining argument? basic skill list:
		
		// level portion
		if (min_level != -1 && max_level != -1) {
			sprintf(lbuf, "level %d-%d", min_level, max_level);
		}
		else if (min_level != -1) {
			sprintf(lbuf, "level %d and higher", min_level);
		}
		else if (max_level != -1) {
			sprintf(lbuf, "up to level %d", max_level);
		}
		else {
			sprintf(lbuf, "(skill level %d)", GET_SKILL_LEVEL(ch));
		}
		
		// start string
		build_page_display(ch, "You know the following skills %s:", lbuf);
		
		HASH_ITER(sorted_hh, sorted_skills, skill, next_skill) {
			if (SKILL_FLAGGED(skill, SKILLF_IN_DEVELOPMENT)) {
				continue;
			}
			level = get_skill_level(ch, SKILL_VNUM(skill));
			if (min_level != -1 && level < min_level) {
				continue;
			}
			if (max_level != -1 && level > max_level) {
				continue;
			}
			if (!SKILL_FLAGGED(skill, SKILLF_BASIC) && level < 1 && get_skill_exp(ch, SKILL_VNUM(skill)) < 0.1) {
				continue;	// don't show non-basic skills if the player doesn't have them
			}
			
			// add to data
			CREATE(skdat, struct skill_display_t, 1);
			skdat->level = level;
			skdat->name_ptr = SKILL_NAME(skill);
			skdat->string = str_dup(get_skill_row_display(ch, skill));
			DL_APPEND(skdat_list, skdat);
		}
		
		// sort if needed?
		if (sort_level) {
			DL_SORT(skdat_list, sort_skill_display_by_level);
		}
		else if (sort_alpha) {
			DL_SORT(skdat_list, sort_skill_display_by_name);
		}
		
		// build display
		any = FALSE;
		DL_FOREACH(skdat_list, skdat) {
			any = TRUE;
			build_page_display_str(ch, skdat->string);
		}
		
		free_skill_display_t(skdat_list);
		skdat_list = NULL;
		
		// orphaned abilities (ones with parents), commands only
		*lbuf = '\0';
		HASH_ITER(hh, GET_ABILITY_HASH(ch), plab, next_plab) {
			abil = plab->ptr;
			
			// ALWAYS use current set for tehse abilities
			if (!plab->purchased[GET_CURRENT_SKILL_SET(ch)]) {
				continue;
			}
			if (!ABIL_COMMAND(abil)) {
				continue;	// only showing commands here
			}
			if (!has_ability_data_any(abil, ADL_PARENT)) {
				continue;	// only looking for abilities with parents
			}
			
			safe_snprintf(lbuf + strlen(lbuf), sizeof(lbuf) - strlen(lbuf), "%s%s%s\t0", *lbuf ? ", ": "", ability_color(ch, abil), ABIL_NAME(abil));
		}
		HASH_ITER(hh, GET_BONUS_ABILITIES(ch), bonus_abil, next_bonus_abil) {
			if (!(abil = ability_proto(bonus_abil->vnum))) {
				continue;	// no ability?
			}
			
			// show it
			safe_snprintf(lbuf + strlen(lbuf), sizeof(lbuf) - strlen(lbuf), "%s%s%s\t0", *lbuf ? ", ": "", ability_color(ch, abil), ABIL_NAME(abil));
		}
		if (*lbuf) {
			build_page_display(ch, "Other abilities: %s\r\n", lbuf);
		}
		
		// footer
		if (!any) {
			build_page_display_str(ch, " none");
		}
		build_page_display_str(ch, get_skill_gain_display(ch));
		send_page_display(ch);
	}
	else if (!str_cmp(arg, "buy")) {
		// purchase
		if (!*arg2) {
			msg_to_char(ch, "Usage: skills buy <ability>\r\n");
			msg_to_char(ch, "You can see a list of abilities for each skill by typing 'skills <skill>'.\r\n");
			return;
		}
		
		if (!(abil = find_ability_by_name(arg2))) {
			// did they try to skill-buy a skill?
			if ((skill = find_skill_by_name(arg2))) {
				msg_to_char(ch, "You can only \"skill buy\" an ability from one of your skill trees, not the whole %s tree.\r\n", skill->name);
			}
			else {
				msg_to_char(ch, "No such ability '%s'.\r\n", arg2);
			}
			return;
		}
		
		if (has_ability(ch, ABIL_VNUM(abil))) {
			msg_to_char(ch, "You already know the %s ability.\r\n", ABIL_NAME(abil));
			return;
		}
		
		if (!ABIL_IS_PURCHASE(abil) && !IS_IMMORTAL(ch)) {
			msg_to_char(ch, "%s: You cannot buy that ability.\r\n", ABIL_NAME(abil));
			return;
		}
		
		if (ABIL_ASSIGNED_SKILL(abil) && get_ability_points_available_for_char(ch, SKILL_VNUM(ABIL_ASSIGNED_SKILL(abil))) < 1 && !IS_IMMORTAL(ch)) {
			msg_to_char(ch, "You have no points available to spend in %s.\r\n", SKILL_NAME(ABIL_ASSIGNED_SKILL(abil)));
			return;
		}
		
		// check level
		if (ABIL_ASSIGNED_SKILL(abil) && get_skill_level(ch, SKILL_VNUM(ABIL_ASSIGNED_SKILL(abil))) < ABIL_SKILL_LEVEL(abil)) {
			msg_to_char(ch, "You need at least %d in %s to buy %s.\r\n", ABIL_SKILL_LEVEL(abil), SKILL_NAME(ABIL_ASSIGNED_SKILL(abil)), ABIL_NAME(abil));
			return;
		}
		
		// check pre-req
		if (ABIL_ASSIGNED_SKILL(abil) && (skab = find_skill_ability(ABIL_ASSIGNED_SKILL(abil), abil)) && skab->prerequisite != NO_PREREQ && !has_ability(ch, skab->prerequisite)) {
			msg_to_char(ch, "You need to buy %s before you can buy %s.\r\n", get_ability_name_by_vnum(skab->prerequisite), ABIL_NAME(abil));
			return;
		}
		
		// good to go
		add_ability(ch, abil, FALSE);
		msg_to_char(ch, "You purchase %s.\r\n", ABIL_NAME(abil));
		queue_delayed_update(ch, CDU_SAVE);
		
		update_class_and_abilities(ch);
	}
	else if (!str_cmp(arg, "reset")) {
		// self-clear!
		if (!*arg2) {
			msg_to_char(ch, "Usage: skills reset <skill>\r\n");
			msg_to_char(ch, "You have free resets available in: ");
			
			found = FALSE;
			HASH_ITER(sorted_hh, sorted_skills, skill, next_skill) {
				if (SKILL_FLAGGED(skill, SKILLF_IN_DEVELOPMENT)) {
					continue;
				}
				
				if (get_skill_resets(ch, SKILL_VNUM(skill)) > 0) {
					msg_to_char(ch, "%s%s", (found ? ", " : ""), SKILL_NAME(skill));
					if (get_skill_resets(ch, SKILL_VNUM(skill)) > 1) {
						msg_to_char(ch, " (%d)", get_skill_resets(ch, SKILL_VNUM(skill)));
					}
					found = TRUE;
				}
			}
			
			if (found) {
				msg_to_char(ch, "\r\n");
			}
			else {
				msg_to_char(ch, "none\r\n");
			}
		}
		else if (!(skill = find_skill_by_name(arg2))) {
			msg_to_char(ch, "No such skill %s.\r\n", arg2);
		}
		else if (!IS_IMMORTAL(ch) && get_skill_resets(ch, SKILL_VNUM(skill)) == 0) {
			msg_to_char(ch, "You do not have a free reset available for %s.\r\n", SKILL_NAME(skill));
		}
		else {
			if (!IS_IMMORTAL(ch) && (skdata = get_skill_data(ch, SKILL_VNUM(skill), TRUE))) {
				skdata->resets = MAX(skdata->resets - 1, 0);
			}
			clear_char_abilities(ch, SKILL_VNUM(skill));
			update_class_and_abilities(ch);
			
			msg_to_char(ch, "You have reset your %s abilities.\r\n", SKILL_NAME(skill));
			queue_delayed_update(ch, CDU_SAVE | CDU_MSDP_SKILLS);
		}
		
		// end "reset"
		return;
	}
	else if (!str_cmp(arg, "drop")) {
		strcpy(lbuf, arg2);
		half_chop(lbuf, arg, arg2);	// "skill" "level"
		
		if (!*arg || !*arg2) {
			msg_to_char(ch, "Warning: This command reduces your level in a skill.\r\n");
			msg_to_char(ch, "Usage: skill drop <skill> <0/%d/%d>\r\n", BASIC_SKILL_CAP, SPECIALTY_SKILL_CAP);
		}
		else if (!(skill = find_skill_by_name(arg))) {
			if (find_ability_by_name(arg)) {
				msg_to_char(ch, "You cannot drop abilities with this command. Use \"skill reset\" instead.\r\n");
			}
			else {
				msg_to_char(ch, "Unknown skill '%s'.\r\n", arg);
			}
		}
		else if (SKILL_MIN_DROP_LEVEL(skill) >= MAX_SKILL_CAP) {
			msg_to_char(ch, "You can't drop your skill level in %s.\r\n", SKILL_NAME(skill));
		}
		else if (!is_number(arg2)) {
			msg_to_char(ch, "Invalid level.\r\n");
		}
		else if ((level = atoi(arg2)) >= get_skill_level(ch, SKILL_VNUM(skill))) {
			msg_to_char(ch, "You can only drop skills to lower levels.\r\n");
		}
		else if (level < SKILL_MIN_DROP_LEVEL(skill)) {
			msg_to_char(ch, "You can't drop %s lower than %d.\r\n", SKILL_NAME(skill), SKILL_MIN_DROP_LEVEL(skill));
		}
		else if (level != SKILL_MIN_DROP_LEVEL(skill) && level != BASIC_SKILL_CAP && level != SPECIALTY_SKILL_CAP) {
			if (SKILL_MIN_DROP_LEVEL(skill) < BASIC_SKILL_CAP) {
				msg_to_char(ch, "You can only drop %s to the following levels: %d, %d, %d\r\n", SKILL_NAME(skill), SKILL_MIN_DROP_LEVEL(skill), BASIC_SKILL_CAP, SPECIALTY_SKILL_CAP);
			}
			else if (SKILL_MIN_DROP_LEVEL(skill) < SPECIALTY_SKILL_CAP) {
				msg_to_char(ch, "You can only drop %s to the following levels: %d, %d\r\n", SKILL_NAME(skill), SKILL_MIN_DROP_LEVEL(skill), SPECIALTY_SKILL_CAP);
			}
			else {
				msg_to_char(ch, "You can only drop %s to the following levels: %d\r\n", SKILL_NAME(skill), SKILL_MIN_DROP_LEVEL(skill));
			}
		}
		else {
			// good to go!
			msg_to_char(ch, "You have dropped your %s skill to %d and reset abilities above that level.\r\n", SKILL_NAME(skill), level);
			set_skill(ch, SKILL_VNUM(skill), level);
			check_un_vampire(ch, FALSE);
			update_class_and_abilities(ch);
			check_ability_levels(ch, SKILL_VNUM(skill));
			
			queue_delayed_update(ch, CDU_SAVE | CDU_MSDP_SKILLS);
		}
	}
	else if (!str_cmp(arg, "swap")) {
		if (IS_IMMORTAL(ch)) {
			perform_swap_skill_sets(ch);
			msg_to_char(ch, "You swap skill sets.\r\n");
		}
		else if (!config_get_bool("skill_swap_allowed")) {
			msg_to_char(ch, "This game does not allow skill swap.\r\n");
		}
		else if (GET_SKILL_LEVEL(ch) < config_get_int("skill_swap_min_level")) {
			msg_to_char(ch, "You require skill level %d to swap skill sets.\r\n", config_get_int("skill_swap_min_level"));
		}
		else if (GET_POS(ch) < POS_STANDING) {
			msg_to_char(ch, "You can't swap skill sets right now.\r\n");
		}
		else if (GET_ACTION(ch) == ACT_SWAP_SKILL_SETS) {
			msg_to_char(ch, "You're already doing that. Type 'stop' to cancel.\r\n");
		}
		else if (GET_ACTION(ch) != ACT_NONE) {
			msg_to_char(ch, "You're too busy to swap skill sets right now.\r\n");
		}
		else {
			start_action(ch, ACT_SWAP_SKILL_SETS, 3);
			act("You prepare to swap skill sets...", FALSE, ch, NULL, NULL, TO_CHAR);
			act("$n prepares to swap skill sets...", TRUE, ch, NULL, NULL, TO_ROOM);
		}
	}
	else if (is_abbrev(arg, "synergy") && strlen(arg) >= 3) {
		strcpy(lbuf, arg2);
		ptr = one_word(lbuf, arg);	// arg: 1st skill
		one_word(ptr, arg2);	// arg2: 2nd skill
		
		// validate args
		if (!*arg || !*arg2) {
			msg_to_char(ch, "Usage: skill synergy <skill 1> <skill 2>\r\n");
			return;
		}
		if (!(synergy[0] = find_skill(arg)) || SKILL_FLAGGED(synergy[0], SKILLF_IN_DEVELOPMENT)) {
			msg_to_char(ch, "No such skill '%s'.\r\n", arg);
			return;
		}
		if (!(synergy[1] = find_skill(arg2)) || SKILL_FLAGGED(synergy[1], SKILLF_IN_DEVELOPMENT)) {
			msg_to_char(ch, "No such skill '%s'.\r\n", arg2);
			return;
		}
		
		// check if I have the skill, if needed
		for (iter = 0; iter < 2; ++iter) {
			if (SKILL_FLAGGED(synergy[iter], SKILLF_NO_PREVIEW) && get_skill_level(ch, SKILL_VNUM(synergy[iter])) < 1 && !IS_IMMORTAL(ch)) {
				msg_to_char(ch, "You don't know the %s skill.\r\n", SKILL_NAME(synergy[iter]));
				return;
			}
		}
		
		// displays: first build the list of abils
		for (iter = 0; iter < 2; ++iter) {
			skill_data *first = synergy[iter], *second = synergy[iter ? 0 : 1];
			
			LL_FOREACH(SKILL_SYNERGIES(first), syn) {
				if (syn->skill != SKILL_VNUM(second)) {
					continue;
				}
				
				add_synergy_display_ability(&sdt_list, syn->role, SKILL_VNUM(first), SKILL_MAX_LEVEL(first), SKILL_VNUM(second), syn->level, syn->ability);
			}
		}
		
		// and show it...
		LL_SORT(sdt_list, sort_sdt_list);
		any = FALSE;
		
		msg_to_char(ch, "Synergy abilities:\r\n");
		LL_FOREACH(sdt_list, sdt) {
			if (PRF_FLAGGED(ch, PRF_SCREEN_READER)) {
				msg_to_char(ch, " %d %s/%d %s %s: ", sdt->level[0], get_skill_abbrev_by_vnum(sdt->skill[0]), sdt->level[1], get_skill_abbrev_by_vnum(sdt->skill[1]), sdt->role == NOTHING ? "(all)" : class_role[sdt->role]);
			}
			else {	// non-screenreader
				msg_to_char(ch, " %s%d %s/%d %s %s\t0: ", sdt->role == NOTHING ? "\t0" : class_role_color[sdt->role], sdt->level[0], get_skill_abbrev_by_vnum(sdt->skill[0]), sdt->level[1], get_skill_abbrev_by_vnum(sdt->skill[1]), sdt->role == NOTHING ? "(all)" : class_role[sdt->role]);
			}
			
			LL_SORT(sdt->abils, sort_sda_list);
			line = FALSE;
			LL_FOREACH(sdt->abils, sda) {
				msg_to_char(ch, "%s%s%s\t0", line ? ", " : "", has_ability(ch, sda->vnum) ? "\tg" : "", get_ability_name_by_vnum(sda->vnum));
				line = TRUE;
				any = TRUE;
			}
			
			msg_to_char(ch, "\r\n");
		}
		
		// done
		if (!any) {
			msg_to_char(ch, " none\r\n");
		}
		
		free_synergy_display_list(sdt_list);
	}
	else if (IS_IMMORTAL(ch) && !str_cmp(arg, "sell")) {
		// purchase		
		if (!*arg2) {
			msg_to_char(ch, "Usage: skills sell <ability>\r\n");
			msg_to_char(ch, "You can see a list of abilities for each skill by typing 'skills <skill>'.\r\n");
			return;
		}
		
		if (!(abil = find_ability_by_name(arg2))) {
			msg_to_char(ch, "No such ability '%s'.\r\n", arg2);
			return;
		}
		
		if (!has_ability(ch, ABIL_VNUM(abil))) {
			msg_to_char(ch, "You do not know the %s ability.\r\n", ABIL_NAME(abil));
			return;
		}
		
		// good to go
		msg_to_char(ch, "You no longer know %s.\r\n", ABIL_NAME(abil));

		remove_ability(ch, abil, FALSE);
		check_skill_sell(ch, abil);
		queue_delayed_update(ch, CDU_SAVE);
		update_class_and_abilities(ch);
	}
	else if (!find_skill_or_ability_for_command(arg, arg2, whole_arg, &skill, &abil)) {
		msg_to_char(ch, "No such skill or ability.\r\n");
	}
	else if (skill) {
		// show 1 skill's details
		
		// validate
		if (SKILL_FLAGGED(skill, SKILLF_NO_PREVIEW) && get_skill_level(ch, SKILL_VNUM(skill)) < 1 && !IS_IMMORTAL(ch)) {
			msg_to_char(ch, "You don't know the %s skill.\r\n", SKILL_NAME(skill));
			return;
		}
		
		// if no levels were requested, default to only showing up to the next cap
		if (min_level == -1 && max_level == -1 && !want_all) {
			max_level = NEXT_CAP_LEVEL(ch, SKILL_VNUM(skill));
			show_all_info = TRUE;
		}
		
		// header
		build_page_display_str(ch, get_skill_row_display(ch, skill));
		
		points = get_ability_points_available_for_char(ch, SKILL_VNUM(skill));
		if (points > 0 && !PRF_FLAGGED(ch, PRF_NO_TUTORIALS)) {
			build_page_display(ch, "You have %d ability point%s to spend. Type 'skill buy <ability>' to purchase a new ability.", points, (points != 1 ? "s" : ""));
		}
		
		// list
		get_skill_abilities_display(&skdat_list, ch, skill, NO_PREREQ, (sort_level || sort_alpha || min_level != -1) ? -1 : 1, min_level, max_level);
		
		// sort if needed?
		if (sort_level) {
			DL_SORT(skdat_list, sort_skill_display_by_level);
		}
		else if (sort_alpha) {
			DL_SORT(skdat_list, sort_skill_display_by_name);
		}
		
		// build display
		DL_FOREACH(skdat_list, skdat) {
			if (min_level != -1 && skdat->level < min_level) {
				continue;
			}
			if (max_level != -1 && skdat->level > max_level) {
				continue;
			}
			
			// show it
			build_page_display_str(ch, skdat->string);
		}
		
		free_skill_display_t(skdat_list);
		skdat_list = NULL;
		
		if (show_all_info && !PRF_FLAGGED(ch, PRF_NO_TUTORIALS) && max_level < SKILL_MAX_LEVEL(skill)) {
			build_page_display(ch, "(For higher level abilities, use 'skill %s -all')", SKILL_NAME(skill));
		}
		
		send_page_display(ch);
	}
	else if (abil) {
		// ability details
		show_ability_info(ch, abil, NULL, TRUE);
	}
	else {	// this should be unreachable
		msg_to_char(ch, "No such skill or ability.\r\n");
	}
}


// this is a temporary command for picking skill specs, and will ultimately be replaced by quests or other mechanisms
ACMD(do_specialize) {
	struct player_skill_data *plsk, *next_plsk;
	int count, max_level_allowed, specialty_allowed;
	skill_data *sk;
	
	skip_spaces(&argument);
	
	// make certain these are up-to-date
	max_level_allowed = config_get_int("skills_at_max_level");
	specialty_allowed = config_get_int("skills_at_specialty_level") + max_level_allowed;
	
	if (!*argument) {
		msg_to_char(ch, "Specialize in which skill?\r\n");
	}
	else if (!(sk = find_skill_by_name(argument))) {
		msg_to_char(ch, "No such skill.\r\n");
	}
	else if (SKILL_FLAGGED(sk, SKILLF_NO_SPECIALIZE)) {
		msg_to_char(ch, "You can't specialize in that skill using this command.\r\n");
	}
	else if (get_skill_level(ch, SKILL_VNUM(sk)) != BASIC_SKILL_CAP && get_skill_level(ch, SKILL_VNUM(sk)) != SPECIALTY_SKILL_CAP) {
		msg_to_char(ch, "You can only specialize skills which are at %d or %d.\r\n", BASIC_SKILL_CAP, SPECIALTY_SKILL_CAP);
	}
	else if (get_skill_level(ch, SKILL_VNUM(sk)) >= SKILL_MAX_LEVEL(sk)) {
		msg_to_char(ch, "%s cannot go above %d.\r\n", SKILL_NAME(sk), SKILL_MAX_LEVEL(sk));
	}
	else {
		// check > basic
		if (get_skill_level(ch, SKILL_VNUM(sk)) == BASIC_SKILL_CAP) {
			count = 0;
			HASH_ITER(hh, GET_SKILL_HASH(ch), plsk, next_plsk) {
				if (plsk->level > BASIC_SKILL_CAP) {
					++count;
				}
			}
			if ((count + 1) > specialty_allowed) {
				msg_to_char(ch, "You can only have at most %d skill%s above level %d.\r\n", specialty_allowed, (specialty_allowed != 1 ? "s" : ""), BASIC_SKILL_CAP);
				return;
			}
		}
		
		// check > specialty
		if (get_skill_level(ch, SKILL_VNUM(sk)) == SPECIALTY_SKILL_CAP) {
			count = 0;
			HASH_ITER(hh, GET_SKILL_HASH(ch), plsk, next_plsk) {
				if (plsk->level > SPECIALTY_SKILL_CAP) {
					++count;
				}
			}
			if ((count + 1) > max_level_allowed) {
				msg_to_char(ch, "You can only have %d skill%s above level %d.\r\n", max_level_allowed, PLURAL(max_level_allowed), SPECIALTY_SKILL_CAP);
				return;
			}
		}

		// le done
		set_skill(ch, SKILL_VNUM(sk), get_skill_level(ch, SKILL_VNUM(sk)) + 1);
		msg_to_char(ch, "You have specialized in %s.\r\n", SKILL_NAME(sk));

		// check class and skill levels
		update_class_and_abilities(ch);
	}
}


 //////////////////////////////////////////////////////////////////////////////
//// HELPERS /////////////////////////////////////////////////////////////////

/**
* Checks if ch can legally gain experience from an ability used against vict.
*
* @param char_data *ch The player trying to gain exp.
* @param char_data *vict The victim of the ability.
* @return bool TRUE if okay to gain experience, or FALSE.
*/
bool can_gain_exp_from(char_data *ch, char_data *vict) {
	if (IS_NPC(ch)) {
		return FALSE;	// mobs gain no exp
	}
	if (ch == vict || !vict) {
		return TRUE;	// always okay
	}
	if (MOB_FLAGGED(vict, MOB_NO_EXPERIENCE)) {
		return FALSE;
	}
	if ((!IS_NPC(vict) || GET_CURRENT_SCALE_LEVEL(vict) > 0) && get_approximate_level(vict) < get_approximate_level(ch) - config_get_int("exp_level_difference")) {
		return FALSE;
	}
	
	// seems ok
	return TRUE;
}


/**
* Determines if the character has the required skill and level to wear an item
* (armor, shield, weapons).
*
* @param char_data *ch The person trying to wear it.
* @param obj_data *item The thing he's trying to wear.
* @param bool send_messages If TRUE, sends its own errors.
* @return bool TRUE if ch can use the item, or FALSE.
*/
bool can_wear_item(char_data *ch, obj_data *item, bool send_messages) {
	char buf[MAX_STRING_LENGTH], part[MAX_STRING_LENGTH];
	any_vnum abil = NO_ABIL, tech = NOTHING;
	struct obj_apply *app;
	int iter, level_min;
	bool honed;

	// players won't be able to use gear >= these levels if their skill level is < the level
	int skill_level_ranges[] = { MAX_SKILL_CAP, SPECIALTY_SKILL_CAP, BASIC_SKILL_CAP, -1 };	// terminate with -1
	
	if (IS_NPC(ch)) {
		return TRUE;
	}
	
	if (IS_ARMOR(item)) {
		switch (GET_ARMOR_TYPE(item)) {
			case ARMOR_MAGE: {
				tech = PTECH_ARMOR_MAGE;
				break;
			}
			case ARMOR_LIGHT: {
				tech = PTECH_ARMOR_LIGHT;
				break;
			}
			case ARMOR_MEDIUM: {
				tech = PTECH_ARMOR_MEDIUM;
				break;
			}
			case ARMOR_HEAVY: {
				tech = PTECH_ARMOR_HEAVY;
				break;
			}
		}
	}
	else if (IS_MISSILE_WEAPON(item)) {
		tech = PTECH_RANGED_COMBAT;
	}
	else if (IS_SHIELD(item)) {
		tech = PTECH_BLOCK;
	}
	
	if (abil != NO_ABIL && !has_ability(ch, abil)) {
		if (send_messages) {
			safe_snprintf(buf, sizeof(buf), "You require the %s ability to use $p.", get_ability_name_by_vnum(abil));
			act(buf, FALSE, ch, item, NULL, TO_CHAR);
		}
		return FALSE;
	}
	if (tech != NOTHING && !has_player_tech(ch, tech)) {
		if (send_messages) {
			// build a list of abilities that offer this tech
			if (ability_string_for_player_tech(ch, tech, part, sizeof(part))) {
				safe_snprintf(buf, sizeof(buf), "You don't have the correct ability to use $p (%s).", part);
				act(buf, FALSE, ch, item, NULL, TO_CHAR);
			}
			else {
				act("You don't have the correct ability to use $p.", FALSE, ch, item, NULL, TO_CHAR);
			}
		}
		return FALSE;
	}
	
	// check honed
	honed = FALSE;
	LL_FOREACH(GET_OBJ_APPLIES(item), app) {
		if (app->apply_type == APPLY_TYPE_HONED) {
			honed = TRUE;
		}
	}
	if (honed && !has_player_tech(ch, PTECH_USE_HONED_GEAR)) {
		if (send_messages) {
			act("You don't have the ability to use honed gear like $p.", FALSE, ch, item, NULL, TO_CHAR);
		}
		return FALSE;
	}
	
	// check levels
	if (!IS_IMMORTAL(ch)) {
		if (GET_OBJ_CURRENT_SCALE_LEVEL(item) <= MAX_SKILL_CAP) {
			for (iter = 0; skill_level_ranges[iter] != -1; ++iter) {
				if (GET_OBJ_CURRENT_SCALE_LEVEL(item) > skill_level_ranges[iter] && GET_SKILL_LEVEL(ch) < skill_level_ranges[iter]) {
					if (send_messages) {
						safe_snprintf(buf, sizeof(buf), "You need to be skill level %d to use $p.", skill_level_ranges[iter]);
						act(buf, FALSE, ch, item, NULL, TO_CHAR);
					}
					return FALSE;
				}
			}
		}
		else {	// > 100
			if (OBJ_FLAGGED(item, OBJ_BIND_ON_PICKUP)) {
				level_min = GET_OBJ_CURRENT_SCALE_LEVEL(item) - 50;
			}
			else {
				level_min = GET_OBJ_CURRENT_SCALE_LEVEL(item) - 25;
			}
			level_min = MAX(level_min, MAX_SKILL_CAP);
			if (GET_SKILL_LEVEL(ch) < MAX_SKILL_CAP) {
				if (send_messages) {
					safe_snprintf(buf, sizeof(buf), "You need to be skill level %d and total level %d to use $p.", MAX_SKILL_CAP, level_min);
					act(buf, FALSE, ch, item, NULL, TO_CHAR);
				}
				return FALSE;
			}
			if (GET_HIGHEST_KNOWN_LEVEL(ch) < level_min) {
				if (send_messages) {
					safe_snprintf(buf, sizeof(buf), "You need to be level %d to use $p.", level_min);
					act(buf, FALSE, ch, item, NULL, TO_CHAR);
				}
				return FALSE;
			}
		}
	}
	
	return TRUE;
}


/**
* Audits skills on startup. Erroring skills are set IN-DEVELOPMENT
*/
void check_skills(void) {
	struct synergy_ability *syn, *next_syn;
	struct skill_ability *skab, *next_skab;
	skill_data *skill, *next_skill;
	bool error;
	
	HASH_ITER(hh, skill_table, skill, next_skill) {
		error = FALSE;
		LL_FOREACH_SAFE(SKILL_ABILITIES(skill), skab, next_skab) {
			if (!find_ability_by_vnum(skab->vnum)) {
				log("- Skill [%d] %s has invalid ability %d", SKILL_VNUM(skill), SKILL_NAME(skill), skab->vnum);
				error = TRUE;
				LL_DELETE(SKILL_ABILITIES(skill), skab);
				free(skab);
			}
		}
		LL_FOREACH_SAFE(SKILL_SYNERGIES(skill), syn, next_syn) {
			if (!find_ability_by_vnum(syn->ability) || !find_skill_by_vnum(syn->skill)) {
				log("- Skill [%d] %s has invalid synergy skill/ability %d/%d", SKILL_VNUM(skill), SKILL_NAME(skill), syn->skill, syn->ability);
				error = TRUE;
				LL_DELETE(SKILL_SYNERGIES(skill), syn);
				free(syn);
			}
		}
		
		if (error) {
			SET_BIT(SKILL_FLAGS(skill), SKILLF_IN_DEVELOPMENT);
		}
		
		// ensure sort
		LL_SORT(SKILL_ABILITIES(skill), sort_skill_abilities);
		LL_SORT(SKILL_SYNERGIES(skill), sort_synergies);
	}
}


/**
* Finds a particular ability's skill assignment data, if any.
*
* @param skill_data *skill The skill whose assignments to check.
* @param ability_data *abil The ability we're looking for.
* @return struct skill_ability* The skill_ability entry, or NULL if none.
*/
struct skill_ability *find_skill_ability(skill_data *skill, ability_data *abil) {
	struct skill_ability *find;
	
	if (!skill || !abil) {
		return NULL;
	}
	
	LL_SEARCH_SCALAR(SKILL_ABILITIES(skill), find, vnum, ABIL_VNUM(abil));
	return find;
}


/**
* Finds an assigned skill for an ability OR one of those ability's parents.
*
* @param ability_data *abil Which ability to find a skill for.
* @param char_data *ch Optional: Only consider abilities possessed by this person (NULL to skip this).
* @return skill_data* The skill if found, or NULL if not.
*/
skill_data *find_assigned_skill(ability_data *abil, char_data *ch) {
	ability_data *find;
	struct ability_data_list *adl;
	
	// my own skill
	if (ABIL_ASSIGNED_SKILL(abil)) {
		return ABIL_ASSIGNED_SKILL(abil);
	}
	
	// check parents
	LL_FOREACH(ABIL_DATA(abil), adl) {
		if (adl->type == ADL_PARENT && (!ch || has_ability(ch, adl->vnum)) && (find = ability_proto(adl->vnum)) && ABIL_ASSIGNED_SKILL(find)) {
			return ABIL_ASSIGNED_SKILL(find);
		}
	}
	
	// no?
	return NULL;
}


/**
* Finds a skill by ambiguous argument, which may be a vnum or a name.
* Names are matched by exact match first, or by multi-abbrev.
*
* @param char *argument The user input.
* @return skill_data* The skill, or NULL if it doesn't exist.
*/
skill_data *find_skill(char *argument) {
	skill_data *skill;
	any_vnum vnum;
	
	if (isdigit(*argument) && (vnum = atoi(argument)) >= 0 && (skill = find_skill_by_vnum(vnum))) {
		return skill;
	}
	else {
		return find_skill_by_name(argument);
	}
}


/**
* Look up a skill by multi-abbrev, preferring exact matches. Won't find in-dev
* skills.
*
* @param char *name The skill name to look up.
* @param bool allow_abbrev If TRUE, allows abbreviations.
* @return skill_data* The skill, or NULL if it doesn't exist.
*/
skill_data *find_skill_by_name_exact(char *name, bool allow_abbrev) {
	skill_data *skill, *next_skill, *partial = NULL;
	
	if (!*name) {
		return NULL;	// shortcut
	}
	
	HASH_ITER(sorted_hh, sorted_skills, skill, next_skill) {
		if (SKILL_FLAGGED(skill, SKILLF_IN_DEVELOPMENT)) {
			continue;
		}
		
		// matches:
		if (!str_cmp(name, SKILL_NAME(skill)) || !str_cmp(name, SKILL_ABBREV(skill))) {
			// perfect match
			return skill;
		}
		if (allow_abbrev && !partial && is_multiword_abbrev(name, SKILL_NAME(skill))) {
			// probable match
			partial = skill;
		}
	}
	
	// no exact match...
	return partial;
}


/**
* @param any_vnum vnum Any skill vnum
* @return skill_data* The skill, or NULL if it doesn't exist
*/
skill_data *find_skill_by_vnum(any_vnum vnum) {
	skill_data *skill;
	
	if (vnum < 0 || vnum == NOTHING) {
		return NULL;
	}
	
	HASH_FIND_INT(skill_table, &vnum, skill);
	return skill;
}


/**
* @param any_vnum vnum A skill vnum.
* @return char* The skill abbreviation, or "???" if no match.
*/
char *get_skill_abbrev_by_vnum(any_vnum vnum) {
	skill_data *skill = find_skill_by_vnum(vnum);
	return skill ? SKILL_ABBREV(skill) : "???";
}


/**
* @param any_vnum vnum A skill vnum.
* @return char* The skill name, or "Unknown" if no match.
*/
char *get_skill_name_by_vnum(any_vnum vnum) {
	skill_data *skill = find_skill_by_vnum(vnum);
	return skill ? SKILL_NAME(skill) : "Unknown";
}


/**
* Ensures the player has all level-zero abilities. These are abilities that all
* players always have.
*
* @param char_data *ch The character to give the abilitiies to.
*/
void give_level_zero_abilities(char_data *ch) {
	ability_data *abil, *next_abil;
	int set;
	
	if (IS_NPC(ch)) {
		return;
	}
	
	HASH_ITER(hh, ability_table, abil, next_abil) {
		if (!ABIL_ASSIGNED_SKILL(abil)) {
			continue;	// must be assigned
		}
		if (SKILL_FLAGGED(ABIL_ASSIGNED_SKILL(abil), SKILLF_IN_DEVELOPMENT)) {
			continue;	// skip in-dev skills
		}
		if (ABIL_SKILL_LEVEL(abil) > 0) {
			continue;	// must be level-0
		}
		
		// ensure the character has a skill entry for the skill
		get_skill_data(ch, SKILL_VNUM(ABIL_ASSIGNED_SKILL(abil)), TRUE);
		
		// check that the player has it
		for (set = 0; set < NUM_SKILL_SETS; ++set) {
			if (!has_ability_in_set(ch, ABIL_VNUM(abil), set)) {
				// add the ability for this set
				add_ability_by_set(ch, abil, set, FALSE);
			}
		}
	}
}


/**
* @param char_data *ch
* @return bool TRUE if the character is somewhere he can cook, else FALSE
*/
bool has_cooking_fire(char_data *ch) {
	obj_data *obj;
	
	if (has_player_tech(ch, PTECH_LIGHT_FIRE)) {
		return TRUE;
	}

	if (room_has_function_and_city_ok(GET_LOYALTY(ch), IN_ROOM(ch), FNC_COOKING_FIRE)) {	
		return TRUE;
	}
	
	// fires in room
	DL_FOREACH2(ROOM_CONTENTS(IN_ROOM(ch)), obj, next_content) {
		if (LIGHT_IS_LIT(obj) && LIGHT_FLAGGED(obj, LIGHT_FLAG_COOKING_FIRE)) {
			return TRUE;
		}
	}
	
	// fires in inventory
	DL_FOREACH2(ch->carrying, obj, next_content) {
		if (LIGHT_IS_LIT(obj) && LIGHT_FLAGGED(obj, LIGHT_FLAG_COOKING_FIRE)) {
			return TRUE;
		}
	}
	
	return FALSE;
}


/* tie/untie an npc */
void perform_npc_tie(char_data *ch, char_data *victim, int subcmd) {
	bool kept = FALSE;
	obj_data *rope;

	if (!subcmd && MOB_FLAGGED(victim, MOB_TIED))
		act("$E's already tied up!", FALSE, ch, 0, victim, TO_CHAR);
	else if (subcmd && !MOB_FLAGGED(victim, MOB_TIED))
		act("$E isn't even tied up!", FALSE, ch, 0, victim, TO_CHAR);
	else if (MOB_FLAGGED(victim, MOB_TIED)) {
		act("You untie $N.", FALSE, ch, 0, victim, TO_CHAR);
		act("$n unties you!", FALSE, ch, 0, victim, TO_VICT | TO_SLEEP);
		act("$n unties $N.", FALSE, ch, 0, victim, TO_NOTVICT);
		remove_mob_flags(victim, MOB_TIED);
		
		if (GET_ROPE_VNUM(victim) != NOTHING && (rope = read_object(GET_ROPE_VNUM(victim), TRUE))) {
			obj_to_char(rope, ch);
			scale_item_to_level(rope, 1);	// minimum scale
			if (load_otrigger(rope)) {
				act("You receive $p.", FALSE, ch, rope, NULL, TO_CHAR);
			}
		}
		GET_ROPE_VNUM(victim) = NOTHING;
		request_char_save_in_world(victim);
	}
	else if (!MOB_FLAGGED(victim, MOB_ANIMAL)) {
		msg_to_char(ch, "You can only tie animals.\r\n");
	}
	else if (!(rope = get_component_in_list(COMP_ROPE, ch->carrying, &kept))) {
		msg_to_char(ch, "You don't seem to have any rope%s.\r\n", kept ? " that isn't marked 'keep'" : "");
	}
	else if (ROOM_SECT_FLAGGED(IN_ROOM(ch), SECTF_FRESH_WATER | SECTF_OCEAN))
		msg_to_char(ch, "You can't tie it here.\r\n");
	else {
		act("You tie $N up.", FALSE, ch, 0, victim, TO_CHAR);
		act("$n ties you up.", FALSE, ch, 0, victim, TO_VICT | TO_SLEEP);
		act("$n ties $N up.", FALSE, ch, 0, victim, TO_NOTVICT);
		set_mob_flags(victim, MOB_TIED);
		GET_ROPE_VNUM(victim) = GET_OBJ_VNUM(rope);
		extract_obj(rope);
		request_char_save_in_world(victim);
	}
}


/**
* Counts the words of text in a skill's strings.
*
* @param skill_data *skill The skill whose strings to count.
* @return int The number of words in the skill's strings.
*/
int wordcount_skill(skill_data *skill) {
	int count = 0;
	
	count += wordcount_string(SKILL_NAME(skill));
	count += wordcount_string(SKILL_ABBREV(skill));
	count += wordcount_string(SKILL_DESC(skill));
	
	return count;
}


 //////////////////////////////////////////////////////////////////////////////
//// UTILITIES ///////////////////////////////////////////////////////////////

/**
* Checks for common skill problems and reports them to ch.
*
* @param skill_data *skill The item to audit.
* @param char_data *ch The person to report to.
* @return bool TRUE if any problems were reported; FALSE if all good.
*/
bool audit_skill(skill_data *skill, char_data *ch) {
	struct skill_ability *skab, *find;
	skill_data *iter, *next_iter;
	bool problem = FALSE;
	ability_data *abil;
	
	if (SKILL_FLAGGED(skill, SKILLF_IN_DEVELOPMENT)) {
		olc_audit_msg(ch, SKILL_VNUM(skill), "IN-DEVELOPMENT");
		problem = TRUE;
	}
	if (!SKILL_NAME(skill) || !*SKILL_NAME(skill) || !str_cmp(SKILL_NAME(skill), default_skill_name)) {
		olc_audit_msg(ch, SKILL_VNUM(skill), "No name set");
		problem = TRUE;
	}
	if (!SKILL_ABBREV(skill) || !*SKILL_ABBREV(skill) || !str_cmp(SKILL_ABBREV(skill), default_skill_abbrev)) {
		olc_audit_msg(ch, SKILL_VNUM(skill), "No abbrev set");
		problem = TRUE;
	}
	if (!SKILL_DESC(skill) || !*SKILL_DESC(skill) || !str_cmp(SKILL_DESC(skill), default_skill_desc)) {
		olc_audit_msg(ch, SKILL_VNUM(skill), "No description set");
		problem = TRUE;
	}
	
	// ability assignments
	LL_FOREACH(SKILL_ABILITIES(skill), skab) {
		if (!(abil = find_ability_by_vnum(skab->vnum))) {
			olc_audit_msg(ch, SKILL_VNUM(skill), "Invalid ability %d", skab->vnum);
			problem = TRUE;
			continue;
		}
		
		if (is_class_ability(abil)) {
			olc_audit_msg(ch, SKILL_VNUM(skill), "Ability %d %s is a class ability", ABIL_VNUM(abil), ABIL_NAME(abil));
			problem = TRUE;
		}
		
		// verify tree
		if (skab->prerequisite != NO_ABIL) {
			if (skab->vnum == skab->prerequisite) {
				olc_audit_msg(ch, SKILL_VNUM(skill), "Ability %d %s is its own prerequisite", ABIL_VNUM(abil), ABIL_NAME(abil));
				problem = TRUE;
			}
			
			LL_SEARCH_SCALAR(SKILL_ABILITIES(skill), find, vnum, skab->prerequisite);
			if (!find) {
				olc_audit_msg(ch, SKILL_VNUM(skill), "Ability %d %s has missing prerequisite", ABIL_VNUM(abil), ABIL_NAME(abil));
				problem = TRUE;
			}
			else if (skab->level < find->level) {
				olc_audit_msg(ch, SKILL_VNUM(skill), "Ability %d %s is lower level than its prerequisite", ABIL_VNUM(abil), ABIL_NAME(abil));
				problem = TRUE;
			}
		}
	}
	
	// other skills
	HASH_ITER(hh, skill_table, iter, next_iter) {
		if (iter == skill || SKILL_VNUM(iter) == SKILL_VNUM(skill)) {
			continue;
		}
		
		if (!str_cmp(SKILL_NAME(iter), SKILL_NAME(skill))) {
			olc_audit_msg(ch, SKILL_VNUM(skill), "Same name as skill %d", SKILL_VNUM(iter));
			problem = TRUE;
		}
		
		// ensure no abilities are assigned to both
		LL_FOREACH(SKILL_ABILITIES(skill), skab) {
			LL_SEARCH_SCALAR(SKILL_ABILITIES(iter), find, vnum, skab->vnum);
			if (find && (abil = find_ability_by_vnum(skab->vnum))) {
				olc_audit_msg(ch, SKILL_VNUM(skill), "Ability %d %s is also assigned to skill %d %s", ABIL_VNUM(abil), ABIL_NAME(abil), SKILL_VNUM(iter), SKILL_NAME(iter));
				problem = TRUE;
			}
		}
	}
	
	return problem;
}


/**
* For the .list command.
*
* @param skill_data *skill The thing to list.
* @param bool detail If TRUE, provide additional details
* @return char* The line to show (without a CRLF).
*/
char *list_one_skill(skill_data *skill, bool detail) {
	static char output[MAX_STRING_LENGTH];
	
	if (detail) {
		safe_snprintf(output, sizeof(output), "[%5d] %s - %s", SKILL_VNUM(skill), SKILL_NAME(skill), SKILL_DESC(skill));
	}
	else {
		safe_snprintf(output, sizeof(output), "[%5d] %s", SKILL_VNUM(skill), SKILL_NAME(skill));
	}
		
	return output;
}


/**
* Searches properties of skills.
*
* @param char_data *ch The person searching.
* @param char *argument The argument they entered.
*/
void olc_fullsearch_skill(char_data *ch, char *argument) {
	bool any;
	char type_arg[MAX_INPUT_LENGTH], val_arg[MAX_INPUT_LENGTH], find_keywords[MAX_INPUT_LENGTH];
	int count;
	bitvector_t only_flags = NOBITS, not_flagged = NOBITS;
	int vmin = NOTHING, vmax = NOTHING, only_level = NOTHING, level_over = NOTHING, level_under = NOTHING;
	int only_mindrop = NOTHING, mindrop_over = NOTHING, mindrop_under = NOTHING;
	skill_data *sk, *next_sk;
	
	if (!*argument) {
		msg_to_char(ch, "See HELP QEDIT FULLSEARCH for syntax.\r\n");
		return;
	}
	
	// process argument
	*find_keywords = '\0';
	while (*argument) {
		// figure out a type
		argument = any_one_arg(argument, type_arg);
		
		if (!strcmp(type_arg, "-")) {
			continue;	// just skip stray dashes
		}
		
		FULLSEARCH_FLAGS("flags", only_flags, skill_flags)
		FULLSEARCH_FLAGS("flagged", only_flags, skill_flags)
		FULLSEARCH_FLAGS("unflagged", not_flagged, skill_flags)
		FULLSEARCH_INT("vmin", vmin, 0, INT_MAX)
		FULLSEARCH_INT("vmax", vmax, 0, INT_MAX)
		FULLSEARCH_INT("maxlevel", only_level, 0, INT_MAX)
		FULLSEARCH_INT("maxlevelover", level_over, 0, INT_MAX)
		FULLSEARCH_INT("maxlevelunder", level_under, 0, INT_MAX)
		FULLSEARCH_INT("mindrop", only_mindrop, 0, INT_MAX)
		FULLSEARCH_INT("mindropover", mindrop_over, 0, INT_MAX)
		FULLSEARCH_INT("mindropunder", mindrop_under, 0, INT_MAX)
		
		else {	// not sure what to do with it? treat it like a keyword
			sprintf(find_keywords + strlen(find_keywords), "%s%s", *find_keywords ? " " : "", type_arg);
		}
		
		// prepare for next loop
		skip_spaces(&argument);
	}
	
	build_page_display(ch, "Skill fullsearch: %s", show_color_codes(find_keywords));
	count = 0;
	
	// okay now look up skills
	HASH_ITER(hh, skill_table, sk, next_sk) {
		if ((vmin != NOTHING && SKILL_VNUM(sk) < vmin) || (vmax != NOTHING && SKILL_VNUM(sk) > vmax)) {
			continue;	// vnum range
		}
		if (not_flagged != NOBITS && IS_SET(SKILL_FLAGS(sk), not_flagged)) {
			continue;
		}
		if (only_flags != NOBITS && (SKILL_FLAGS(sk) & only_flags) != only_flags) {
			continue;
		}
		if (only_level != NOTHING && SKILL_MAX_LEVEL(sk) != only_level) {
			continue;
		}
		if (level_over != NOTHING && SKILL_MAX_LEVEL(sk) < level_over) {
			continue;
		}
		if (level_under != NOTHING && SKILL_MAX_LEVEL(sk) > level_over) {
			continue;
		}
		if (only_mindrop != NOTHING && SKILL_MIN_DROP_LEVEL(sk) != only_mindrop) {
			continue;
		}
		if (mindrop_over != NOTHING && SKILL_MIN_DROP_LEVEL(sk) < mindrop_over) {
			continue;
		}
		if (mindrop_under != NOTHING && SKILL_MIN_DROP_LEVEL(sk) > mindrop_over) {
			continue;
		}
		
		// search strings
		if (*find_keywords) {
			any = FALSE;
			if (multi_isname(find_keywords, SKILL_NAME(sk))) {
				any = TRUE;
			}
			else if (multi_isname(find_keywords, SKILL_DESC(sk))) {
				any = TRUE;
			}
			else if (multi_isname(find_keywords, SKILL_ABBREV(sk))) {
				any = TRUE;
			}
			
			// did we find a match in any string
			if (!any) {
				continue;
			}
		}
		
		// show it
		build_page_display(ch, "[%5d] %s", SKILL_VNUM(sk), SKILL_NAME(sk));
		++count;
	}
	
	if (count > 0) {
		build_page_display(ch, "(%d skills)", count);
	}
	else {
		build_page_display_str(ch, " none");
	}
	
	send_page_display(ch);
}


/**
* Searches for all uses of a skill and displays them.
*
* @param char_data *ch The player.
* @param any_vnum vnum The skill vnum.
*/
void olc_search_skill(char_data *ch, any_vnum vnum) {
	skill_data *skill = find_skill_by_vnum(vnum), *sk, *next_sk;
	archetype_data *arch, *next_arch;
	quest_data *quest, *next_quest;
	progress_data *prg, *next_prg;
	struct archetype_skill *arsk;
	struct class_skill_req *clsk;
	struct synergy_ability *syn;
	social_data *soc, *next_soc;
	trig_data *trig, *next_trig;
	class_data *cls, *next_cls;
	int found;
	bool any;
	
	if (!skill) {
		msg_to_char(ch, "There is no skill %d.\r\n", vnum);
		return;
	}
	
	found = 0;
	build_page_display(ch, "Occurrences of skill %d (%s):", vnum, SKILL_NAME(skill));
	
	// archetypes
	HASH_ITER(hh, archetype_table, arch, next_arch) {
		LL_FOREACH(GET_ARCH_SKILLS(arch), arsk) {
			if (arsk->skill == vnum) {
				++found;
				build_page_display(ch, "ARCH [%5d] %s", GET_ARCH_VNUM(arch), GET_ARCH_NAME(arch));
				break;	// only need 1
			}
		}
	}
	
	// classes
	HASH_ITER(hh, class_table, cls, next_cls) {
		LL_FOREACH(CLASS_SKILL_REQUIREMENTS(cls), clsk) {
			if (clsk->vnum == vnum) {
				++found;
				build_page_display(ch, "CLS [%5d] %s", CLASS_VNUM(cls), CLASS_NAME(cls));
				break;	// only need 1
			}
		}
	}
	
	// progress
	HASH_ITER(hh, progress_table, prg, next_prg) {
		// REQ_x: requirement search
		any = find_requirement_in_list(PRG_TASKS(prg), REQ_SKILL_LEVEL_OVER, vnum);
		any |= find_requirement_in_list(PRG_TASKS(prg), REQ_SKILL_LEVEL_UNDER, vnum);
		any |= find_requirement_in_list(PRG_TASKS(prg), REQ_CAN_GAIN_SKILL, vnum);
		
		if (any) {
			++found;
			build_page_display(ch, "PRG [%5d] %s", PRG_VNUM(prg), PRG_NAME(prg));
		}
	}
	
	// quests
	HASH_ITER(hh, quest_table, quest, next_quest) {
		any = find_quest_reward_in_list(QUEST_REWARDS(quest), QR_SET_SKILL, vnum);
		any |= find_quest_reward_in_list(QUEST_REWARDS(quest), QR_SKILL_EXP, vnum);
		any |= find_quest_reward_in_list(QUEST_REWARDS(quest), QR_SKILL_LEVELS, vnum);
		any |= find_requirement_in_list(QUEST_TASKS(quest), REQ_SKILL_LEVEL_OVER, vnum);
		any |= find_requirement_in_list(QUEST_PREREQS(quest), REQ_SKILL_LEVEL_OVER, vnum);
		any |= find_requirement_in_list(QUEST_TASKS(quest), REQ_SKILL_LEVEL_UNDER, vnum);
		any |= find_requirement_in_list(QUEST_PREREQS(quest), REQ_SKILL_LEVEL_UNDER, vnum);
		any |= find_requirement_in_list(QUEST_TASKS(quest), REQ_CAN_GAIN_SKILL, vnum);
		any |= find_requirement_in_list(QUEST_PREREQS(quest), REQ_CAN_GAIN_SKILL, vnum);
		
		if (any) {
			++found;
			build_page_display(ch, "QST [%5d] %s", QUEST_VNUM(quest), QUEST_NAME(quest));
		}
	}
	
	// skills
	HASH_ITER(hh, skill_table, sk, next_sk) {
		LL_FOREACH(SKILL_SYNERGIES(sk), syn) {
			if (syn->skill == vnum) {
				++found;
				build_page_display(ch, "SKL [%5d] %s", SKILL_VNUM(sk), SKILL_NAME(sk));
				break;
			}
		}
	}
	
	// socials
	HASH_ITER(hh, social_table, soc, next_soc) {
		any = find_requirement_in_list(SOC_REQUIREMENTS(soc), REQ_SKILL_LEVEL_OVER, vnum);
		any |= find_requirement_in_list(SOC_REQUIREMENTS(soc), REQ_SKILL_LEVEL_UNDER, vnum);
		any |= find_requirement_in_list(SOC_REQUIREMENTS(soc), REQ_CAN_GAIN_SKILL, vnum);
		
		if (any) {
			++found;
			build_page_display(ch, "SOC [%5d] %s", SOC_VNUM(soc), SOC_NAME(soc));
		}
	}
	
	// triggers
	HASH_ITER(hh, trigger_table, trig, next_trig) {
		if (trigger_has_link(trig, OLC_SKILL, vnum)) {
			++found;
			build_page_display(ch, "TRG [%5d] %s", GET_TRIG_VNUM(trig), GET_TRIG_NAME(trig));
		}
	}
	
	if (found > 0) {
		build_page_display(ch, "%d location%s shown", found, PLURAL(found));
	}
	else {
		build_page_display_str(ch, " none");
	}
	
	send_page_display(ch);
}


/**
* Delete by vnum: synergy abilities
*
* @param struct synergy_ability **list Pointer to a linked list of synergies.
* @param any_vnum vnum The vnum to delete from that list.
* @return bool TRUE if it deleted at least one, FALSE if it wasn't in the list.
*/
bool remove_ability_from_synergy_abilities(struct synergy_ability **list, any_vnum abil_vnum) {
	struct synergy_ability *iter, *next_iter;
	bool found = FALSE;
	
	LL_FOREACH_SAFE(*list, iter, next_iter) {
		if (iter->ability == abil_vnum) {
			LL_DELETE(*list, iter);
			free(iter);
			found = TRUE;
		}
	}
	
	return found;
}


/**
* Delete by vnum: synergy skills
*
* @param struct synergy_ability **list Pointer to a linked list of synergies.
* @param any_vnum vnum The vnum to delete from that list.
* @return bool TRUE if it deleted at least one, FALSE if it wasn't in the list.
*/
bool remove_skill_from_synergy_abilities(struct synergy_ability **list, any_vnum skill_vnum) {
	struct synergy_ability *iter, *next_iter;
	bool found = FALSE;
	
	LL_FOREACH_SAFE(*list, iter, next_iter) {
		if (iter->skill == skill_vnum) {
			LL_DELETE(*list, iter);
			free(iter);
			found = TRUE;
		}
	}
	
	return found;
}


/**
* Delete by vnum: skill abilities
*
* @param struct skill_ability **list Pointer to a linked list of skill abilities.
* @param any_vnum vnum The vnum to delete from that list.
* @return bool TRUE if it deleted at least one, FALSE if it wasn't in the list.
*/
bool remove_vnum_from_skill_abilities(struct skill_ability **list, any_vnum vnum) {
	struct skill_ability *iter, *next_iter;
	bool found = FALSE;
	
	LL_FOREACH_SAFE(*list, iter, next_iter) {
		if (iter->vnum == vnum) {
			LL_DELETE(*list, iter);
			free(iter);
			found = TRUE;
		}
		else if (iter->prerequisite == vnum) {
			iter->prerequisite = NOTHING;	// drop prereq
			found = TRUE;
		}
	}
	
	return found;
}


/**
* Sorts skill abilities alphabetically.
*
* @param struct skill_ability *a First element.
* @param struct skill_ability *b Second element.
* @return int sort code
*/
int sort_skill_abilities(struct skill_ability *a, struct skill_ability *b) {
	ability_data *a_abil, *b_abil;
	
	a_abil = find_ability_by_vnum(a->vnum);
	b_abil = find_ability_by_vnum(b->vnum);
	
	if (a_abil && b_abil) {
		return strcmp(ABIL_NAME(a_abil), ABIL_NAME(b_abil));
	}
	else if (a_abil) {
		return -1;
	}
	else {
		return 1;
	}
}


// Simple vnum sorter for the skill hash
int sort_skills(skill_data *a, skill_data *b) {
	return SKILL_VNUM(a) - SKILL_VNUM(b);
}


// typealphabetic sorter for sorted_skills
int sort_skills_by_data(skill_data *a, skill_data *b) {
	return strcmp(NULLSAFE(SKILL_NAME(a)), NULLSAFE(SKILL_NAME(b)));
}


/**
* Sorts skill synergies by role/skill/level.
*
* @param struct synergy_ability *a First element.
* @param struct synergy_ability *b Second element.
* @return int sort code
*/
int sort_synergies(struct synergy_ability *a, struct synergy_ability *b) {
	if (a->role != b->role) {
		return (a->role - b->role);
	}
	else if (a->skill != b->skill) {
		return (a->skill - b->skill);
	}
	else {
		return (b->level - a->level);
	}
}


 //////////////////////////////////////////////////////////////////////////////
//// DATABASE ////////////////////////////////////////////////////////////////

/**
* Puts a skill into the hash table.
*
* @param skill_data *skill The skill data to add to the table.
*/
void add_skill_to_table(skill_data *skill) {
	skill_data *find;
	any_vnum vnum;
	
	if (skill) {
		vnum = SKILL_VNUM(skill);
		HASH_FIND_INT(skill_table, &vnum, find);
		if (!find) {
			HASH_ADD_INT(skill_table, vnum, skill);
			HASH_SORT(skill_table, sort_skills);
		}
		
		// sorted table
		HASH_FIND(sorted_hh, sorted_skills, &vnum, sizeof(int), find);
		if (!find) {
			HASH_ADD(sorted_hh, sorted_skills, vnum, sizeof(int), skill);
			HASH_SRT(sorted_hh, sorted_skills, sort_skills_by_data);
		}
	}
}


/**
* Removes a skill from the hash table.
*
* @param skill_data *skill The skill data to remove from the table.
*/
void remove_skill_from_table(skill_data *skill) {
	HASH_DEL(skill_table, skill);
	HASH_DELETE(sorted_hh, sorted_skills, skill);
}


/**
* Initializes a new skill. This clears all memory for it, so set the vnum
* AFTER.
*
* @param skill_data *skill The skill to initialize.
*/
void clear_skill(skill_data *skill) {
	memset((char *) skill, 0, sizeof(skill_data));
	
	SKILL_VNUM(skill) = NOTHING;
	SKILL_MAX_LEVEL(skill) = MAX_SKILL_CAP;
}


/**
* Duplicates a list of skill abilities, for editing.
*
* @param struct skill_ability *input The head of the list to copy.
* @return struct skill_ability* The copied list.
*/
struct skill_ability *copy_skill_abilities(struct skill_ability *input) {
	struct skill_ability *el, *iter, *list = NULL;
	
	LL_FOREACH(input, iter) {
		CREATE(el, struct skill_ability, 1);
		*el = *iter;
		LL_APPEND(list, el);
	}
	
	return list;
}


/**
* Duplicates a list of skill synergies, for editing.
*
* @param struct synergy_ability *input The head of the list to copy.
* @return struct synergy_ability* The copied list.
*/
struct synergy_ability *copy_synergy_abilities(struct synergy_ability *input) {
	struct synergy_ability *el, *iter, *list = NULL;
	
	LL_FOREACH(input, iter) {
		CREATE(el, struct synergy_ability, 1);
		*el = *iter;
		LL_APPEND(list, el);
	}
	
	return list;
}


/**
* @param struct skill_ability *list Frees the memory for this list.
*/
void free_skill_abilities(struct skill_ability *list) {
	struct skill_ability *tmp, *next;
	
	LL_FOREACH_SAFE(list, tmp, next) {
		free(tmp);
	}
}


/**
* @param struct synergy_ability *list Frees the memory for this list.
*/
void free_synergy_abilities(struct synergy_ability *list) {
	struct synergy_ability *tmp, *next;
	
	LL_FOREACH_SAFE(list, tmp, next) {
		free(tmp);
	}
}


/**
* frees up memory for a skill data item.
*
* See also: olc_delete_skill
*
* @param skill_data *skill The skill data to free.
*/
void free_skill(skill_data *skill) {
	skill_data *proto = find_skill_by_vnum(SKILL_VNUM(skill));
	
	if (SKILL_NAME(skill) && (!proto || SKILL_NAME(skill) != SKILL_NAME(proto))) {
		free(SKILL_NAME(skill));
	}
	if (SKILL_ABBREV(skill) && (!proto || SKILL_ABBREV(skill) != SKILL_ABBREV(proto))) {
		free(SKILL_ABBREV(skill));
	}
	if (SKILL_DESC(skill) && (!proto || SKILL_DESC(skill) != SKILL_DESC(proto))) {
		free(SKILL_DESC(skill));
	}
	if (SKILL_ABILITIES(skill) && (!proto || SKILL_ABILITIES(skill) != SKILL_ABILITIES(proto))) {
		free_skill_abilities(SKILL_ABILITIES(skill));
	}
	if (SKILL_SYNERGIES(skill) && (!proto || SKILL_SYNERGIES(skill) != SKILL_SYNERGIES(proto))) {
		free_synergy_abilities(SKILL_SYNERGIES(skill));
	}
	
	free(skill);
}


/**
* Read one skill from file.
*
* @param FILE *fl The open .skill file
* @param any_vnum vnum The skill vnum
*/
void parse_skill(FILE *fl, any_vnum vnum) {
	char line[256], error[256], str_in[256];
	struct skill_ability *skabil;
	struct synergy_ability *syn;
	skill_data *skill, *find;
	int int_in[5];
	
	CREATE(skill, skill_data, 1);
	clear_skill(skill);
	SKILL_VNUM(skill) = vnum;
	
	HASH_FIND_INT(skill_table, &vnum, find);
	if (find) {
		log("WARNING: Duplicate skill vnum #%d", vnum);
		// but have to load it anyway to advance the file
	}
	add_skill_to_table(skill);
		
	// for error messages
	sprintf(error, "skill vnum %d", vnum);
	
	// lines 1-3: string
	SKILL_NAME(skill) = fread_string(fl, error);
	SKILL_ABBREV(skill) = fread_string(fl, error);
	SKILL_DESC(skill) = fread_string(fl, error);
	
	// line 4: flags
	if (!get_line(fl, line) || sscanf(line, "%s", str_in) != 1) {
		log("SYSERR: Format error in line 4 of %s", error);
		exit(1);
	}
	
	SKILL_FLAGS(skill) = asciiflag_conv(str_in);
		
	// optionals
	for (;;) {
		if (!get_line(fl, line)) {
			log("SYSERR: Format error in %s, expecting alphabetic flags", error);
			exit(1);
		}
		switch (*line) {
			case 'A': {	// ability assignment
				if (sscanf(line, "A %d %d %d", &int_in[0], &int_in[1], &int_in[2]) != 3) {
					log("SYSERR: Format error in A line of %s", error);
					exit(1);
				}
				
				CREATE(skabil, struct skill_ability, 1);
				skabil->vnum = int_in[0];
				skabil->prerequisite = int_in[1];
				skabil->level = int_in[2];
				
				LL_APPEND(SKILL_ABILITIES(skill), skabil);
				break;
			}
			
			case 'L': {	// additional level data
				if (sscanf(line, "L %d %d", &int_in[0], &int_in[1]) != 2) {
					log("SYSERR: Format error in L line of %s", error);
					exit(1);
				}
				
				SKILL_MAX_LEVEL(skill) = int_in[0];
				SKILL_MIN_DROP_LEVEL(skill) = int_in[1];
				break;
			}
			
			case 'Y': {	// synergies
				if (sscanf(line, "Y %d %d %d %d %d\n", &int_in[0], &int_in[1], &int_in[2], &int_in[3], &int_in[4]) != 5) {
					log("SYSERR: Format error in Y line of %s", error);
					exit(1);
				}
				
				CREATE(syn, struct synergy_ability, 1);
				syn->role = int_in[0];
				syn->skill = int_in[1];
				syn->level = int_in[2];
				syn->ability = int_in[3];
				syn->unused = int_in[4];
				
				LL_APPEND(SKILL_SYNERGIES(skill), syn);
				break;
			}
			
			// end
			case 'S': {
				return;
			}
			
			default: {
				log("SYSERR: Format error in %s, expecting alphabetic flags", error);
				exit(1);
			}
		}
	}
}


// writes entries in the skill index
void write_skill_index(FILE *fl) {
	skill_data *skill, *next_skill;
	int this, last;
	
	last = NO_WEAR;
	HASH_ITER(hh, skill_table, skill, next_skill) {
		// determine "zone number" by vnum
		this = (int)(SKILL_VNUM(skill) / 100);
	
		if (this != last) {
			fprintf(fl, "%d%s\n", this, SKILL_SUFFIX);
			last = this;
		}
	}
}


/**
* Outputs one skill in the db file format, starting with a #VNUM and
* ending with an S.
*
* @param FILE *fl The file to write it to.
* @param skill_data *skill The thing to save.
*/
void write_skill_to_file(FILE *fl, skill_data *skill) {
	struct synergy_ability *syn;
	struct skill_ability *iter;
	
	if (!fl || !skill) {
		syslog(SYS_ERROR, LVL_START_IMM, TRUE, "SYSERR: write_skill_to_file called without %s", !fl ? "file" : "skill");
		return;
	}
	
	fprintf(fl, "#%d\n", SKILL_VNUM(skill));
	
	// 1-3: strings
	fprintf(fl, "%s~\n", NULLSAFE(SKILL_NAME(skill)));
	fprintf(fl, "%s~\n", NULLSAFE(SKILL_ABBREV(skill)));
	fprintf(fl, "%s~\n", NULLSAFE(SKILL_DESC(skill)));
	
	// 4: flags
	fprintf(fl, "%s\n", bitv_to_alpha(SKILL_FLAGS(skill)));
	
	// A: abilities
	LL_FOREACH(SKILL_ABILITIES(skill), iter) {
		fprintf(fl, "A %d %d %d\n", iter->vnum, iter->prerequisite, iter->level);
	}
	
	// L: additional level data
	fprintf(fl, "L %d %d\n", SKILL_MAX_LEVEL(skill), SKILL_MIN_DROP_LEVEL(skill));
	
	// Y: synergies
	LL_FOREACH(SKILL_SYNERGIES(skill), syn) {
		fprintf(fl, "Y %d %d %d %d %d\n", syn->role, syn->skill, syn->level, syn->ability, syn->unused);
	}
	
	// end
	fprintf(fl, "S\n");
}


 //////////////////////////////////////////////////////////////////////////////
//// OLC HANDLERS ////////////////////////////////////////////////////////////


/**
* Creates a new skill entry.
* 
* @param any_vnum vnum The number to create.
* @return skill_data* The new skill's prototype.
*/
skill_data *create_skill_table_entry(any_vnum vnum) {
	skill_data *skill;
	
	// sanity
	if (find_skill_by_vnum(vnum)) {
		log("SYSERR: Attempting to insert skill at existing vnum %d", vnum);
		return find_skill_by_vnum(vnum);
	}
	
	CREATE(skill, skill_data, 1);
	clear_skill(skill);
	SKILL_VNUM(skill) = vnum;
	SKILL_NAME(skill) = str_dup(default_skill_name);
	SKILL_ABBREV(skill) = str_dup(default_skill_abbrev);
	SKILL_DESC(skill) = str_dup(default_skill_desc);
	add_skill_to_table(skill);

	// save index and skill file now
	save_index(DB_BOOT_SKILL);
	save_library_file_for_vnum(DB_BOOT_SKILL, vnum);

	return skill;
}


/**
* WARNING: This function actually deletes a skill.
*
* @param char_data *ch The person doing the deleting.
* @param any_vnum vnum The vnum to delete.
*/
void olc_delete_skill(char_data *ch, any_vnum vnum) {
	struct player_skill_data *plsk, *next_plsk;
	struct archetype_skill *arsk, *next_arsk;
	archetype_data *arch, *next_arch;
	ability_data *abil, *next_abil;
	quest_data *quest, *next_quest;
	progress_data *prg, *next_prg;
	social_data *soc, *next_soc;
	class_data *cls, *next_cls;
	trig_data *trig, *next_trig;
	descriptor_data *desc;
	skill_data *skill, *sk, *next_sk;
	char_data *chiter;
	char name[256];
	bool found;
	
	if (!(skill = find_skill_by_vnum(vnum))) {
		msg_to_char(ch, "There is no such skill %d.\r\n", vnum);
		return;
	}
	
	safe_snprintf(name, sizeof(name), "%s", NULLSAFE(SKILL_NAME(skill)));
	
	// remove it from the hash table first
	remove_skill_from_table(skill);
	
	// remove from live abilities
	HASH_ITER(hh, ability_table, abil, next_abil) {
		if (ABIL_ASSIGNED_SKILL(abil) == skill) {
			ABIL_ASSIGNED_SKILL(abil) = NULL;
			ABIL_SKILL_LEVEL(abil) = 0;
			// this is not saved data, no need to save library files
		}
	}
	
	// remove from archetypes
	HASH_ITER(hh, archetype_table, arch, next_arch) {
		found = FALSE;
		LL_FOREACH_SAFE(GET_ARCH_SKILLS(arch), arsk, next_arsk) {
			if (arsk->skill == vnum) {
				LL_DELETE(GET_ARCH_SKILLS(arch), arsk);
				free(arsk);
				found |= TRUE;
			}
		}
		
		if (found) {
			SET_BIT(GET_ARCH_FLAGS(arch), ARCH_IN_DEVELOPMENT);
			syslog(SYS_OLC, GET_INVIS_LEV(ch), TRUE, "OLC: Archetype %d %s set IN-DEV due to deleted skill", GET_ARCH_VNUM(arch), GET_ARCH_NAME(arch));
			save_library_file_for_vnum(DB_BOOT_ARCH, GET_ARCH_VNUM(arch));
		}
	}
	
	// remove from classes
	HASH_ITER(hh, class_table, cls, next_cls) {
		found = remove_vnum_from_class_skill_reqs(&CLASS_SKILL_REQUIREMENTS(cls), vnum);
		if (found) {
			SET_BIT(CLASS_FLAGS(cls), CLASSF_IN_DEVELOPMENT);
			syslog(SYS_OLC, GET_INVIS_LEV(ch), TRUE, "OLC: Class %d %s set IN-DEV due to deleted skill", CLASS_VNUM(cls), CLASS_NAME(cls));
			save_library_file_for_vnum(DB_BOOT_CLASS, CLASS_VNUM(cls));
		}
	}
	
	// update progress
	HASH_ITER(hh, progress_table, prg, next_prg) {
		found = delete_requirement_from_list(&PRG_TASKS(prg), REQ_SKILL_LEVEL_OVER, vnum);
		found |= delete_requirement_from_list(&PRG_TASKS(prg), REQ_SKILL_LEVEL_UNDER, vnum);
		found |= delete_requirement_from_list(&PRG_TASKS(prg), REQ_CAN_GAIN_SKILL, vnum);
		
		if (found) {
			SET_BIT(PRG_FLAGS(prg), PRG_IN_DEVELOPMENT);
			syslog(SYS_OLC, GET_INVIS_LEV(ch), TRUE, "OLC: Progress %d %s set IN-DEV due to deleted skill", PRG_VNUM(prg), PRG_NAME(prg));
			save_library_file_for_vnum(DB_BOOT_PRG, PRG_VNUM(prg));
			need_progress_refresh = TRUE;
		}
	}
	
	// remove from quests
	HASH_ITER(hh, quest_table, quest, next_quest) {
		found = delete_quest_reward_from_list(&QUEST_REWARDS(quest), QR_SET_SKILL, vnum);
		found |= delete_quest_reward_from_list(&QUEST_REWARDS(quest), QR_SKILL_EXP, vnum);
		found |= delete_quest_reward_from_list(&QUEST_REWARDS(quest), QR_SKILL_LEVELS, vnum);
		found |= delete_requirement_from_list(&QUEST_TASKS(quest), REQ_SKILL_LEVEL_OVER, vnum);
		found |= delete_requirement_from_list(&QUEST_PREREQS(quest), REQ_SKILL_LEVEL_OVER, vnum);
		found |= delete_requirement_from_list(&QUEST_TASKS(quest), REQ_SKILL_LEVEL_UNDER, vnum);
		found |= delete_requirement_from_list(&QUEST_PREREQS(quest), REQ_SKILL_LEVEL_UNDER, vnum);
		found |= delete_requirement_from_list(&QUEST_TASKS(quest), REQ_CAN_GAIN_SKILL, vnum);
		found |= delete_requirement_from_list(&QUEST_PREREQS(quest), REQ_CAN_GAIN_SKILL, vnum);
		
		if (found) {
			SET_BIT(QUEST_FLAGS(quest), QST_IN_DEVELOPMENT);
			syslog(SYS_OLC, GET_INVIS_LEV(ch), TRUE, "OLC: Quest %d %s set IN-DEV due to deleted skill", QUEST_VNUM(quest), QUEST_NAME(quest));
			save_library_file_for_vnum(DB_BOOT_QST, QUEST_VNUM(quest));
		}
	}
	
	// remove from other skills
	HASH_ITER(hh, skill_table, sk, next_sk) {
		found = remove_skill_from_synergy_abilities(&SKILL_SYNERGIES(sk), vnum);
		if (found) {
			syslog(SYS_OLC, GET_INVIS_LEV(ch), TRUE, "OLC: Skill %d %s lost deleted synergy skill", SKILL_VNUM(sk), SKILL_NAME(sk));
			save_library_file_for_vnum(DB_BOOT_SKILL, SKILL_VNUM(sk));
		}
	}
	
	// remove from socials
	HASH_ITER(hh, social_table, soc, next_soc) {
		found = delete_requirement_from_list(&SOC_REQUIREMENTS(soc), REQ_SKILL_LEVEL_OVER, vnum);
		found |= delete_requirement_from_list(&SOC_REQUIREMENTS(soc), REQ_SKILL_LEVEL_UNDER, vnum);
		found |= delete_requirement_from_list(&SOC_REQUIREMENTS(soc), REQ_CAN_GAIN_SKILL, vnum);
		
		if (found) {
			SET_BIT(SOC_FLAGS(soc), SOC_IN_DEVELOPMENT);
			syslog(SYS_OLC, GET_INVIS_LEV(ch), TRUE, "OLC: Social %d %s set IN-DEV due to deleted skill", SOC_VNUM(soc), SOC_NAME(soc));
			save_library_file_for_vnum(DB_BOOT_SOC, SOC_VNUM(soc));
		}
	}
	
	// update triggers
	HASH_ITER(hh, trigger_table, trig, next_trig) {
		found = trigger_has_link(trig, OLC_SKILL, vnum);
		if (found) {
			syslog(SYS_OLC, GET_INVIS_LEV(ch), TRUE, "OLC: Trigger %d %s lost link to skill [%d] %s", GET_TRIG_VNUM(trig), GET_TRIG_NAME(trig), vnum, name);
			// Doesn't delete
			// save_library_file_for_vnum(DB_BOOT_TRG, GET_TRIG_VNUM(trig));
		}
	}
	
	// remove from live players
	DL_FOREACH2(player_character_list, chiter, next_plr) {
		found = FALSE;
		
		HASH_ITER(hh, GET_SKILL_HASH(chiter), plsk, next_plsk) {
			if (plsk->vnum == vnum) {
				clear_char_abilities(chiter, plsk->vnum);
				HASH_DEL(GET_SKILL_HASH(chiter), plsk);
				free(plsk);
				found = TRUE;
			}
		}
	}
	
	// look for olc editors
	LL_FOREACH(descriptor_list, desc) {
		if (GET_OLC_ARCHETYPE(desc)) {
			found = FALSE;
			LL_FOREACH_SAFE(GET_ARCH_SKILLS(GET_OLC_ARCHETYPE(desc)), arsk, next_arsk) {
				if (arsk->skill == vnum) {
					LL_DELETE(GET_ARCH_SKILLS(GET_OLC_ARCHETYPE(desc)), arsk);
					free(arsk);
					found |= TRUE;
				}
			}
			if (found) {
				msg_to_desc(desc, "A skill has been deleted from the archetype you're editing.\r\n");
			}
		}
		if (GET_OLC_CLASS(desc)) {
			found = remove_vnum_from_class_skill_reqs(&CLASS_SKILL_REQUIREMENTS(GET_OLC_CLASS(desc)), vnum);
			if (found) {
				msg_to_desc(desc, "A skill requirement has been deleted from the class you're editing.\r\n");
			}
		}
		if (GET_OLC_PROGRESS(desc)) {
			found = delete_requirement_from_list(&PRG_TASKS(GET_OLC_PROGRESS(desc)), REQ_SKILL_LEVEL_OVER, vnum);
			found |= delete_requirement_from_list(&PRG_TASKS(GET_OLC_PROGRESS(desc)), REQ_SKILL_LEVEL_UNDER, vnum);
			found |= delete_requirement_from_list(&PRG_TASKS(GET_OLC_PROGRESS(desc)), REQ_CAN_GAIN_SKILL, vnum);
		
			if (found) {
				SET_BIT(QUEST_FLAGS(GET_OLC_PROGRESS(desc)), PRG_IN_DEVELOPMENT);
				msg_to_desc(desc, "A skill requirement has been deleted from the progression goal you're editing.\r\n");
			}
		}
		if (GET_OLC_QUEST(desc)) {
			found = delete_quest_reward_from_list(&QUEST_REWARDS(GET_OLC_QUEST(desc)), QR_SET_SKILL, vnum);
			found |= delete_quest_reward_from_list(&QUEST_REWARDS(GET_OLC_QUEST(desc)), QR_SKILL_EXP, vnum);
			found |= delete_quest_reward_from_list(&QUEST_REWARDS(GET_OLC_QUEST(desc)), QR_SKILL_LEVELS, vnum);
			found |= delete_requirement_from_list(&QUEST_TASKS(GET_OLC_QUEST(desc)), REQ_SKILL_LEVEL_OVER, vnum);
			found |= delete_requirement_from_list(&QUEST_PREREQS(GET_OLC_QUEST(desc)), REQ_SKILL_LEVEL_OVER, vnum);
			found |= delete_requirement_from_list(&QUEST_TASKS(GET_OLC_QUEST(desc)), REQ_SKILL_LEVEL_UNDER, vnum);
			found |= delete_requirement_from_list(&QUEST_PREREQS(GET_OLC_QUEST(desc)), REQ_SKILL_LEVEL_UNDER, vnum);
			found |= delete_requirement_from_list(&QUEST_TASKS(GET_OLC_QUEST(desc)), REQ_CAN_GAIN_SKILL, vnum);
			found |= delete_requirement_from_list(&QUEST_PREREQS(GET_OLC_QUEST(desc)), REQ_CAN_GAIN_SKILL, vnum);
		
			if (found) {
				SET_BIT(QUEST_FLAGS(GET_OLC_QUEST(desc)), QST_IN_DEVELOPMENT);
				msg_to_desc(desc, "A skill used by the quest you are editing was deleted.\r\n");
			}
		}
		if (GET_OLC_SKILL(desc)) {
			found = remove_skill_from_synergy_abilities(&SKILL_SYNERGIES(GET_OLC_SKILL(desc)), vnum);
			if (found) {
				msg_to_desc(desc, "A synergy skill in the skill you are editing was deleted.\r\n");
			}
		}
		if (GET_OLC_SOCIAL(desc)) {
			found = delete_requirement_from_list(&SOC_REQUIREMENTS(GET_OLC_SOCIAL(desc)), REQ_SKILL_LEVEL_OVER, vnum);
			found |= delete_requirement_from_list(&SOC_REQUIREMENTS(GET_OLC_SOCIAL(desc)), REQ_SKILL_LEVEL_UNDER, vnum);
			found |= delete_requirement_from_list(&SOC_REQUIREMENTS(GET_OLC_SOCIAL(desc)), REQ_CAN_GAIN_SKILL, vnum);
		
			if (found) {
				SET_BIT(SOC_FLAGS(GET_OLC_SOCIAL(desc)), SOC_IN_DEVELOPMENT);
				msg_to_desc(desc, "A skill required by the social you are editing was deleted.\r\n");
			}
		}
		if (GET_OLC_TRIGGER(desc)) {
			found = trigger_has_link(GET_OLC_TRIGGER(desc), OLC_SKILL, vnum);
			if (found) {
				msg_to_desc(desc, "Skill [%d] %s was deleted but remains in the link list for the trigger you're editing.", vnum, name);
			}
		}
	}

	// save index and skill file now
	save_index(DB_BOOT_SKILL);
	save_library_file_for_vnum(DB_BOOT_SKILL, vnum);
	
	syslog(SYS_OLC, GET_INVIS_LEV(ch), TRUE, "OLC: %s has deleted skill %d %s", GET_NAME(ch), vnum, name);
	msg_to_char(ch, "Skill %d (%s) deleted.\r\n", vnum, name);
	
	free_skill(skill);
}


/**
* Function to save a player's changes to a skill (or a new one).
*
* @param descriptor_data *desc The descriptor who is saving.
*/
void save_olc_skill(descriptor_data *desc) {
	skill_data *proto, *skill = GET_OLC_SKILL(desc);
	any_vnum vnum = GET_OLC_VNUM(desc);
	UT_hash_handle hh, sorted;
	char_data *ch_iter, *next_ch;

	// have a place to save it?
	if (!(proto = find_skill_by_vnum(vnum))) {
		proto = create_skill_table_entry(vnum);
	}
	
	// free prototype strings and pointers
	if (SKILL_NAME(proto)) {
		free(SKILL_NAME(proto));
	}
	if (SKILL_ABBREV(proto)) {
		free(SKILL_ABBREV(proto));
	}
	if (SKILL_DESC(proto)) {
		free(SKILL_DESC(proto));
	}
	free_skill_abilities(SKILL_ABILITIES(proto));
	free_synergy_abilities(SKILL_SYNERGIES(proto));
	
	// sanity
	if (!SKILL_NAME(skill) || !*SKILL_NAME(skill)) {
		if (SKILL_NAME(skill)) {
			free(SKILL_NAME(skill));
		}
		SKILL_NAME(skill) = str_dup(default_skill_name);
	}
	if (!SKILL_ABBREV(skill) || !*SKILL_ABBREV(skill)) {
		if (SKILL_ABBREV(skill)) {
			free(SKILL_ABBREV(skill));
		}
		SKILL_ABBREV(skill) = str_dup(default_skill_abbrev);
	}
	if (!SKILL_DESC(skill) || !*SKILL_DESC(skill)) {
		if (SKILL_DESC(skill)) {
			free(SKILL_DESC(skill));
		}
		SKILL_DESC(skill) = str_dup(default_skill_desc);
	}
	
	// save data back over the proto-type
	hh = proto->hh;	// save old hash handle
	sorted = proto->sorted_hh;
	*proto = *skill;	// copy over all data
	proto->vnum = vnum;	// ensure correct vnum
	proto->hh = hh;	// restore old hash handle
	proto->sorted_hh = sorted;
		
	// and save to file
	save_library_file_for_vnum(DB_BOOT_SKILL, vnum);

	// ... and update some things
	HASH_SRT(sorted_hh, sorted_skills, sort_skills_by_data);
	read_ability_requirements();
	
	// update all players in case there are new level-0 abilities
	DL_FOREACH_SAFE2(player_character_list, ch_iter, next_ch, next_plr) {
		update_class_and_abilities(ch_iter);
		give_level_zero_abilities(ch_iter);
	}
}


/**
* Creates a copy of a skill, or clears a new one, for editing.
* 
* @param skill_data *input The skill to copy, or NULL to make a new one.
* @return skill_data* The copied skill.
*/
skill_data *setup_olc_skill(skill_data *input) {
	skill_data *new;
	
	CREATE(new, skill_data, 1);
	clear_skill(new);
	
	if (input) {
		// copy normal data
		*new = *input;

		// copy things that are pointers
		SKILL_NAME(new) = SKILL_NAME(input) ? str_dup(SKILL_NAME(input)) : NULL;
		SKILL_ABBREV(new) = SKILL_ABBREV(input) ? str_dup(SKILL_ABBREV(input)) : NULL;
		SKILL_DESC(new) = SKILL_DESC(input) ? str_dup(SKILL_DESC(input)) : NULL;
		SKILL_ABILITIES(new) = copy_skill_abilities(SKILL_ABILITIES(input));
		SKILL_SYNERGIES(new) = copy_synergy_abilities(SKILL_SYNERGIES(input));
	}
	else {
		// brand new: some defaults
		SKILL_NAME(new) = str_dup(default_skill_name);
		SKILL_ABBREV(new) = str_dup(default_skill_abbrev);
		SKILL_DESC(new) = str_dup(default_skill_desc);
		SKILL_FLAGS(new) = SKILLF_IN_DEVELOPMENT;
	}
	
	// done
	return new;	
}


 //////////////////////////////////////////////////////////////////////////////
//// SKILL ABILITY DISPLAY ///////////////////////////////////////////////////

struct skad_element {
	char **text;
	int lines;
	struct skad_element *next;	// linked list
};


/**
* Builds one block of text for a hierarchical ability list.
*
* @param struct skill_ability *list The whole list we're showing part of.
* @param struct skill_ability *parent Which parent ability we are showing.
* @param int indent Number of times to indent this row.
* @param struct skad_element **display A pointer to a list of skad elements, for storing the display.
*/
void get_skad_partial(struct skill_ability *list, struct skill_ability *parent, int indent, struct skad_element **display, struct skad_element *skad) {
	char buf[MAX_STRING_LENGTH];
	struct skill_ability *abil;
	
	LL_FOREACH(list, abil) {
		if (!parent && abil->prerequisite != NO_ABIL) {
			continue;	// only showing freestanding abilities here
		}
		if (parent && abil->prerequisite != parent->vnum) {
			continue;	// wrong sub-tree
		}
		
		// have one to display
		if (!skad) {
			CREATE(skad, struct skad_element, 1);
			skad->lines = 0;
			skad->text = NULL;
			LL_APPEND(*display, skad);
		}
		
		safe_snprintf(buf, sizeof(buf), "%*s%s[%d] %-.17s @ %d", (2 * indent), " ", (parent ? "+ " : ""), abil->vnum, get_ability_name_by_vnum(abil->vnum), abil->level);
		
		// append line
		if (skad->lines > 0) {
			RECREATE(skad->text, char*, skad->lines + 1);
		}
		else {
			CREATE(skad->text, char*, 1);
		}
		skad->text[skad->lines] = str_dup(buf);
		++(skad->lines);
		
		// find any dependent abilities
		get_skad_partial(list, abil, indent + 1, display, skad);
		if (!parent) {
			skad = NULL;	// create new block
		}
	}
}


/**
* Builds the two-column display of abilities for a skill.
*
* @param char_data *for_char Which player is viewing it.
* @param struct skill_ability *list The list to show.
* @param char *save_buffer Text to write the result to.
* @param size_t buflen The max length of save_buffer.
*/
void get_skill_ability_display(char_data *for_char, struct skill_ability *list, char *save_buffer, size_t buflen) {
	struct skad_element *skad, *mid, *display = NULL;
	char **left_text = NULL, **right_text = NULL;
	int left_lines = 0, right_lines = 0;
	int count, iter, total_lines;
	double half;
	size_t size;
	
	// prepare...
	*save_buffer = '\0';
	size = 0;
	
	// fetch set of columns
	get_skad_partial(list, NULL, 0, &display, NULL);
	
	if (!display) {
		return;	// no work
	}
	
	// determine number of blocks per column
	total_lines = 0;
	LL_FOREACH(display, skad) {
		total_lines += skad->lines;
	}
	
	// determine approximate middle
	mid = NULL;
	count = 0;
	half = total_lines / 2.0;
	LL_FOREACH(display, skad) {
		count += skad->lines;
		
		if (count >= (int)round(half)) {
			mid = skad->next;
			break;
		}
	}
	
	// build left column: move string pointers over to new list
	for (skad = display; skad && skad != mid; skad = skad->next) {
		if (left_lines > 0) {
			RECREATE(left_text, char*, left_lines + skad->lines);
		}
		else {
			CREATE(left_text, char*, skad->lines);
		}
		for (iter = 0; iter < skad->lines; ++iter) {
			left_text[left_lines++] = skad->text[iter];
		}
	}
	
	// build right column: move string pointers over to new list
	for (skad = mid; skad; skad = skad->next) {
		if (right_lines > 0) {
			RECREATE(right_text, char*, right_lines + skad->lines);
		}
		else {
			CREATE(right_text, char*, skad->lines);
		}
		for (iter = 0; iter < skad->lines; ++iter) {
			right_text[right_lines++] = skad->text[iter];
		}
	}
	
	// build full display
	if (PRF_FLAGGED(for_char, PRF_SCREEN_READER)) {
		// 1-column version
		for (iter = 0; iter < left_lines; ++iter) {
			size += snprintf(save_buffer + size, buflen - size, " %s\r\n", left_text[iter]);
		}
		for (iter = 0; iter < right_lines; ++iter) {
			size += snprintf(save_buffer + size, buflen - size, " %s\r\n", right_text[iter]);
		}
	}
	else {
		// 2-column version
		iter = 0;
		while (iter < left_lines || iter < right_lines) {
			size += snprintf(save_buffer + size, buflen - size, " %-38.38s", (iter < left_lines ? left_text[iter] : ""));
			if (iter < right_lines) {
				size += snprintf(save_buffer + size, buflen - size, " %-38.38s\r\n", right_text[iter]);
			}
			else {
				size += snprintf(save_buffer + size, buflen - size, "\r\n");
			}
		
			++iter;
		}
	}
	
	// free all the things
	LL_FOREACH_SAFE(display, skad, mid) {
		free(skad);
	}
	for (iter = 0; iter < left_lines; ++iter) {
		free(left_text[iter]);
	}
	for (iter = 0; iter < right_lines; ++iter) {
		free(right_text[iter]);
	}
	free(left_text);
	free(right_text);
}


 //////////////////////////////////////////////////////////////////////////////
//// MAIN DISPLAYS ///////////////////////////////////////////////////////////

/**
* For vstat.
*
* @param char_data *ch The player requesting stats.
* @param skill_data *skill The skill to display.
*/
void do_stat_skill(char_data *ch, skill_data *skill) {
	char part[MAX_STRING_LENGTH];
	struct synergy_ability *syn;
	struct skill_ability *skab;
	int total;
	
	if (!skill) {
		return;
	}
	
	// first line
	build_page_display(ch, "VNum: [\tc%d\t0], Name: \tc%s\t0, Abbrev: \tc%s\t0", SKILL_VNUM(skill), SKILL_NAME(skill), SKILL_ABBREV(skill));
	
	build_page_display(ch, "Description: %s", SKILL_DESC(skill));
	
	build_page_display(ch, "Minimum drop level: [\tc%d\t0], Maximum level: [\tc%d\t0]", SKILL_MIN_DROP_LEVEL(skill), SKILL_MAX_LEVEL(skill));
	
	sprintbit(SKILL_FLAGS(skill), skill_flags, part, TRUE);
	build_page_display(ch, "Flags: \tg%s\t0", part);
	
	LL_COUNT(SKILL_ABILITIES(skill), skab, total);
	build_page_display(ch, "Simplified skill tree: (%d total)", total);
	get_skill_ability_display(ch, SKILL_ABILITIES(skill), part, sizeof(part));
	if (*part) {
		build_page_display_str(ch, part);
	}

	LL_COUNT(SKILL_SYNERGIES(skill), syn, total);
	build_page_display(ch, "Synergy abilities: (%d total)", total);
	get_skill_synergy_display(SKILL_SYNERGIES(skill), part, NULL);
	if (*part) {
		build_page_display_str(ch, part);
	}
	
	send_page_display(ch);
}


/**
* This is the main recipe display for skill OLC. It displays the user's
* currently-edited skill.
*
* @param char_data *ch The person who is editing a skill and will see its display.
*/
void olc_show_skill(char_data *ch) {
	skill_data *skill = GET_OLC_SKILL(ch->desc);
	char lbuf[MAX_STRING_LENGTH];
	struct synergy_ability *syn;
	struct skill_ability *skab;
	int total;
	
	if (!skill) {
		return;
	}
	
	build_page_display(ch, "[%s%d\t0] %s%s\t0", OLC_LABEL_CHANGED, GET_OLC_VNUM(ch->desc), OLC_LABEL_UNCHANGED, !find_skill_by_vnum(SKILL_VNUM(skill)) ? "new skill" : get_skill_name_by_vnum(SKILL_VNUM(skill)));
	build_page_display(ch, "<%sname\t0> %s", OLC_LABEL_STR(SKILL_NAME(skill), default_skill_name), NULLSAFE(SKILL_NAME(skill)));
	build_page_display(ch, "<%sabbrev\t0> %s", OLC_LABEL_STR(SKILL_ABBREV(skill), default_skill_abbrev), NULLSAFE(SKILL_ABBREV(skill)));
	build_page_display(ch, "<%sdescription\t0> %s", OLC_LABEL_STR(SKILL_DESC(skill), default_skill_desc), NULLSAFE(SKILL_DESC(skill)));
	
	sprintbit(SKILL_FLAGS(skill), skill_flags, lbuf, TRUE);
	build_page_display(ch, "<%sflags\t0> %s", OLC_LABEL_VAL(SKILL_FLAGS(skill), SKILLF_IN_DEVELOPMENT), lbuf);
	
	build_page_display(ch, "<%smaxlevel\t0> %d", OLC_LABEL_VAL(SKILL_MAX_LEVEL(skill), MAX_SKILL_CAP), SKILL_MAX_LEVEL(skill));
	build_page_display(ch, "<%smindrop\t0> %d", OLC_LABEL_VAL(SKILL_MIN_DROP_LEVEL(skill), 0), SKILL_MIN_DROP_LEVEL(skill));
	
	LL_COUNT(SKILL_ABILITIES(skill), skab, total);
	build_page_display(ch, "<%stree\t0> %d %s (.showtree to toggle display)", OLC_LABEL_PTR(SKILL_ABILITIES(skill)), total, total == 1 ? "ability" : "abilities");
	if (GET_OLC_SHOW_TREE(ch->desc)) {
		get_skill_ability_display(ch, SKILL_ABILITIES(skill), lbuf, sizeof(lbuf));
		if (*lbuf) {
			build_page_display_str(ch, lbuf);
		}
	}
	
	LL_COUNT(SKILL_SYNERGIES(skill), syn, total);
	build_page_display(ch, "<%ssynergy\t0> %d %s (.showsynergies to toggle display)", OLC_LABEL_PTR(SKILL_SYNERGIES(skill)), total, total == 1 ? "ability" : "abilities");
	if (GET_OLC_SHOW_SYNERGIES(ch->desc)) {
		get_skill_synergy_display(SKILL_SYNERGIES(skill), lbuf, NULL);
		if (*lbuf) {
			build_page_display_str(ch, lbuf);
		}
	}
	
	send_page_display(ch);
}


/**
* Gets the skill synergy display for olc, stat, or other uses.
*
* @param struct synergy_ability *list The list of abilities to display.
* @param char *save_buffer A buffer to store the display to.
* @param char_data *info_ch Optional: highlights abilities this player has (or NULL).
*/
void get_skill_synergy_display(struct synergy_ability *list, char *save_buffer, char_data *info_ch) {
	int count = 0, last_role = -2, last_skill = NOTHING, last_level = -1;
	struct synergy_ability *iter;
	ability_data *abil;
	
	*save_buffer = '\0';
	
	LL_FOREACH(list, iter) {
		if (iter->role != last_role || iter->skill != last_skill || iter->level != last_level) {
			sprintf(save_buffer + strlen(save_buffer), "%s %s%s: [%d] %s %d\t0: ", last_role != -2 ? "\r\n" : "", iter->role == NOTHING ? "\t0" : class_role_color[iter->role], iter->role == NOTHING ? "All roles" : class_role[iter->role], iter->skill, get_skill_name_by_vnum(iter->skill), iter->level);
			last_role = iter->role;
			last_skill = iter->skill;
			last_level = iter->level;
			count = 0;
		}
		
		if ((abil = find_ability_by_vnum(iter->ability))) {
			sprintf(save_buffer + strlen(save_buffer), "%s%s%s\t0", (count++ > 0) ? ", " : "", (info_ch && has_ability(info_ch, iter->ability)) ? "\tg" : "", ABIL_NAME(abil));
		}
		else {
			sprintf(save_buffer + strlen(save_buffer), "%s%d Unknown\t0", (count++ > 0) ? ", " : "", iter->ability);
		}
	}
	
	if (last_role != -2) {
		strcat(save_buffer, "\r\n");
	}
}


/**
* Searches the skill db for a match, and prints it to the character.
*
* @param char *searchname The search string.
* @param char_data *ch The player who is searching.
* @return int The number of matches shown.
*/
int vnum_skill(char *searchname, char_data *ch) {
	skill_data *iter, *next_iter;
	int found = 0;
	
	HASH_ITER(hh, skill_table, iter, next_iter) {
		if (multi_isname(searchname, SKILL_NAME(iter)) || is_abbrev(searchname, SKILL_ABBREV(iter))) {
			build_page_display(ch, "%3d. [%5d] %s", ++found, SKILL_VNUM(iter), SKILL_NAME(iter));
		}
	}
	
	send_page_display(ch);
	return found;
}


 //////////////////////////////////////////////////////////////////////////////
//// OLC MODULES /////////////////////////////////////////////////////////////

OLC_MODULE(skilledit_abbrev) {
	skill_data *skill = GET_OLC_SKILL(ch->desc);
	
	if (color_strlen(argument) != 3) {
		msg_to_char(ch, "Skill abbreviations must be 3 letters.\r\n");
	}
	else if (color_code_length(argument) > 0) {
		msg_to_char(ch, "Skill abbreviations may not contain color codes.\r\n");
	}
	else {
		olc_process_string(ch, argument, "abbreviation", &SKILL_ABBREV(skill));
	}
}


OLC_MODULE(skilledit_description) {
	skill_data *skill = GET_OLC_SKILL(ch->desc);
	olc_process_string(ch, argument, "description", &SKILL_DESC(skill));
}


OLC_MODULE(skilledit_flags) {
	skill_data *skill = GET_OLC_SKILL(ch->desc);
	bool had_indev = IS_SET(SKILL_FLAGS(skill), SKILLF_IN_DEVELOPMENT) ? TRUE : FALSE;
	
	SKILL_FLAGS(skill) = olc_process_flag(ch, argument, "skill", "flags", skill_flags, SKILL_FLAGS(skill));
	
	// validate removal of IN-DEVELOPMENT
	if (had_indev && !IS_SET(SKILL_FLAGS(skill), SKILLF_IN_DEVELOPMENT) && GET_ACCESS_LEVEL(ch) < LVL_UNRESTRICTED_BUILDER && !OLC_FLAGGED(ch, OLC_FLAG_CLEAR_IN_DEV)) {
		msg_to_char(ch, "You don't have permission to remove the IN-DEVELOPMENT flag.\r\n");
		SET_BIT(SKILL_FLAGS(skill), SKILLF_IN_DEVELOPMENT);
	}
}


OLC_MODULE(skilledit_maxlevel) {
	skill_data *skill = GET_OLC_SKILL(ch->desc);
	SKILL_MAX_LEVEL(skill) = olc_process_number(ch, argument, "maximum level", "maxlevel", 1, MAX_SKILL_CAP, SKILL_MAX_LEVEL(skill));
}


OLC_MODULE(skilledit_mindrop) {
	skill_data *skill = GET_OLC_SKILL(ch->desc);
	SKILL_MIN_DROP_LEVEL(skill) = olc_process_number(ch, argument, "minimum drop level", "mindrop", 0, MAX_SKILL_CAP, SKILL_MIN_DROP_LEVEL(skill));
}


OLC_MODULE(skilledit_name) {
	skill_data *skill = GET_OLC_SKILL(ch->desc);
	olc_process_string(ch, argument, "name", &SKILL_NAME(skill));
}


OLC_MODULE(skilledit_showsynergies) {
	if (GET_OLC_SHOW_SYNERGIES(ch->desc)) {
		GET_OLC_SHOW_SYNERGIES(ch->desc) = FALSE;
	}
	else {
		GET_OLC_SHOW_SYNERGIES(ch->desc) = TRUE;
	}
	
	if (PRF_FLAGGED(ch, PRF_NOREPEAT)) {
		send_config_msg(ch, "ok_string");
	}
	else {
		msg_to_char(ch, "Your editor will %s show the synergies.\r\n", GET_OLC_SHOW_SYNERGIES(ch->desc) ? "now" : "no longer");
		olc_show_skill(ch);
	}
}


OLC_MODULE(skilledit_showtree) {
	if (GET_OLC_SHOW_TREE(ch->desc)) {
		GET_OLC_SHOW_TREE(ch->desc) = FALSE;
	}
	else {
		GET_OLC_SHOW_TREE(ch->desc) = TRUE;
	}
	
	if (PRF_FLAGGED(ch, PRF_NOREPEAT)) {
		send_config_msg(ch, "ok_string");
	}
	else {
		msg_to_char(ch, "Your editor will %s show the skill tree.\r\n", GET_OLC_SHOW_TREE(ch->desc) ? "now" : "no longer");
		olc_show_skill(ch);
	}
}


OLC_MODULE(skilledit_synergy) {
	skill_data *skill = GET_OLC_SKILL(ch->desc), *other, *temp;
	char cmd_arg[MAX_INPUT_LENGTH], role_arg[MAX_INPUT_LENGTH], skl_arg[MAX_INPUT_LENGTH], lvl_arg[MAX_INPUT_LENGTH], abil_arg[MAX_INPUT_LENGTH];
	char type_arg[MAX_INPUT_LENGTH], val_arg[MAX_INPUT_LENGTH];
	struct synergy_ability *syn, *next_syn;
	int role = ROLE_NONE, level = 1;
	ability_data *abil;
	bool all, any, all_roles;
	
	argument = any_one_word(argument, role_arg);
	argument = any_one_arg(argument, cmd_arg);
	
	// first args
	all_roles = !str_cmp(role_arg, "all");
	role = all_roles ? NOTHING : search_block(role_arg, class_role, FALSE);
	if (!all_roles && *role_arg && role == NOTHING) {
		msg_to_char(ch, "Invalid role '%s'.\r\n", role_arg);
		return;
	}
	else if (!*cmd_arg) {
		msg_to_char(ch, "Usage: synergy <roll | all> add <skill> <level> <ability>\r\n");
		msg_to_char(ch, "       synergy <role | all> change <ability> <skill | level> <value>\r\n");
		msg_to_char(ch, "       synergy <role | all> remove <ability | all>\r\n");
		return;
	}
	
	// actual commands
	if (is_abbrev(cmd_arg, "add")) {
		argument = any_one_word(argument, skl_arg);
		argument = any_one_arg(argument, lvl_arg);
		skip_spaces(&argument);
		strcpy(abil_arg, argument);
		
		if (!*skl_arg || !*lvl_arg || !*abil_arg) {
			msg_to_char(ch, "Usage: synergy <roll | all> add <skill> <level> <ability>\r\n");
			return;
		}
		else if (!(other = find_skill(skl_arg))) {
			msg_to_char(ch, "Unknown skill '%s'.\r\n", skl_arg);
			return;
		}
		else if (!(abil = find_ability(abil_arg))) {
			msg_to_char(ch, "Unknown ability '%s'.\r\n", abil_arg);
			return;
		}
		else if (ABIL_ASSIGNED_SKILL(abil)) {
			msg_to_char(ch, "You can't assign an ability that is already assigned to a skill tree (%s).\r\n", SKILL_NAME(ABIL_ASSIGNED_SKILL(abil)));
			return;
		}
		else if (has_ability_data_any(abil, ADL_PARENT)) {
			msg_to_char(ch, "You can't assign %s as a synergy ability because it has PARENT data.\r\n", ABIL_NAME(abil));
			return;
		}
		else if (!isdigit(*lvl_arg) || (level = atoi(lvl_arg)) < 1 || level > SKILL_MAX_LEVEL(other)) {
			msg_to_char(ch, "Level must be 1-%d, '%s' given.\r\n", SKILL_MAX_LEVEL(other), lvl_arg);
			return;
		}
		
		// ensure not already on this role
		any = FALSE;
		LL_FOREACH(SKILL_SYNERGIES(skill), syn) {
			if (syn->role == role && syn->skill == SKILL_VNUM(other) && syn->ability == ABIL_VNUM(abil)) {
				any = TRUE;
				break;
			}
		}
		
		if (any) {
			msg_to_char(ch, "It already has %s on the %s role.\r\n", ABIL_NAME(abil), all_roles ? "ALL" : class_role[role]);
		}
		else {
			CREATE(syn, struct synergy_ability, 1);
			syn->role = role;
			syn->skill = SKILL_VNUM(other);
			syn->level = level;
			syn->ability = ABIL_VNUM(abil);
			LL_APPEND(SKILL_SYNERGIES(skill), syn);
			msg_to_char(ch, "You add %s to the %s role when paired with %s %d.\r\n", ABIL_NAME(abil), all_roles ? "ALL" : class_role[role], SKILL_NAME(other), level);
		}
		
		// ensure sorting now
		LL_SORT(SKILL_SYNERGIES(skill), sort_synergies);
	}
	else if (is_abbrev(cmd_arg, "change")) {
		argument = any_one_word(argument, abil_arg);
		argument = one_argument(argument, type_arg);
		argument = any_one_word(argument, val_arg);
		
		if (!*abil_arg) {
			msg_to_char(ch, "Usage: synergy <role | all> change <ability> <skill | level> <value>\r\n");
			return;
		}
		
		abil = find_ability(abil_arg);
		if (!abil) {
			msg_to_char(ch, "Invalid ability '%s'.\r\n", abil_arg);
			return;
		}
		
		// init the data we'll use
		other = NULL;
		level = -1;
		
		// check valid type
		if (is_abbrev(type_arg, "skill")) {
			if (!(other = find_skill(val_arg))) {
				msg_to_char(ch, "Unknown skill '%s'.\r\n", val_arg);
				return;
			}
		}
		else if (is_abbrev(type_arg, "level")) {
			if (!isdigit(*val_arg) || (level = atoi(val_arg)) < 1) {
				msg_to_char(ch, "Level must be 1-%d, '%s' given.\r\n", MAX_SKILL_CAP, val_arg);
				return;
			}
		}
		else {
			msg_to_char(ch, "You can only change the skill or level.\r\n");
			return;
		}
		
		any = FALSE;
		LL_FOREACH_SAFE(SKILL_SYNERGIES(skill), syn, next_syn) {
			if (role != NOTHING && syn->role != role) {
				continue;
			}
			
			if (syn->ability == ABIL_VNUM(abil)) {
				if (other) {
					syn->skill = SKILL_VNUM(other);
				}
				else if (level != -1) {
					if ((temp = find_skill_by_vnum(syn->skill)) && level > SKILL_MAX_LEVEL(temp)) {
						msg_to_char(ch, "Given level %d is above the max level of %s (%d).\r\n", level, SKILL_NAME(temp), SKILL_MAX_LEVEL(temp));
						return;
					}
					else {
						syn->level = level;
					}
				}
				any = TRUE;
			}
		}
		
		if (!any) {
			msg_to_char(ch, "The %s role didn't have %s.\r\n", all_roles ? "ALL" : class_role[role], "that ability");
		}
		else if (other) {
			msg_to_char(ch, "The %s ability on the %s role now comes from %s.\r\n", ABIL_NAME(abil), all_roles ? "ALL" : class_role[role], SKILL_NAME(other));
		}
		else {
			msg_to_char(ch, "The %s ability on the %s role now comes at level %d in the other skill.\r\n", ABIL_NAME(abil), all_roles ? "ALL" : class_role[role], level);
		}
	}
	else if (is_abbrev(cmd_arg, "remove")) {
		skip_spaces(&argument);
		strcpy(abil_arg, argument);
		
		if (!*abil_arg) {
			msg_to_char(ch, "Usage: synergy <role | all> remove <ability | all>\r\n");
			return;
		}
		
		abil = find_ability(abil_arg);
		all = !str_cmp(abil_arg, "all");
		if (!all && !abil) {
			msg_to_char(ch, "Invalid ability '%s'.\r\n", abil_arg);
			return;
		}
		
		any = FALSE;
		LL_FOREACH_SAFE(SKILL_SYNERGIES(skill), syn, next_syn) {
			if (role != NOTHING && syn->role != role) {
				continue;
			}
			if (all && role == NOTHING && syn->role != role) {
				continue;	// prevents 'trying to remove abilities from the ALL role' from removing abilities from all others
			}
			
			if (all || (abil && syn->ability == ABIL_VNUM(abil))) {
				LL_DELETE(SKILL_SYNERGIES(skill), syn);
				free(syn);
				any = TRUE;
			}
		}
		
		if (!any) {
			msg_to_char(ch, "The %s role didn't have %s.\r\n", all_roles ? "ALL" : class_role[role], all ? "any abilities" : "that ability");
		}
		else if (all) {
			msg_to_char(ch, "You remove all abilities from the %s role.\r\n", all_roles ? "ALL" : class_role[role]);
		}
		else {
			msg_to_char(ch, "You remove %s from the %s role.\r\n", ABIL_NAME(abil), all_roles ? "ALL" : class_role[role]);
		}
	}
	else {
		msg_to_char(ch, "Usage: synergy <roll | all> add <skill> <level> <ability>\r\n");
		msg_to_char(ch, "       synergy <role | all> change <ability> <skill | level> <value>\r\n");
		msg_to_char(ch, "       synergy <role | all> remove <ability | all>\r\n");
	}
}


OLC_MODULE(skilledit_tree) {
	skill_data *skill = GET_OLC_SKILL(ch->desc);
	char cmd_arg[MAX_INPUT_LENGTH], abil_arg[MAX_INPUT_LENGTH], sub_arg[MAX_INPUT_LENGTH], req_arg[MAX_INPUT_LENGTH];
	struct skill_ability *skab, *next_skab, *change;
	ability_data *abil = NULL, *requires = NULL;
	bool all = FALSE, found, found_prq;
	int level;
	
	argument = any_one_arg(argument, cmd_arg);
	argument = any_one_word(argument, abil_arg);
	argument = any_one_arg(argument, sub_arg);	// may be level or type
	argument = any_one_word(argument, req_arg);	// may be requires ability or "new value"
	
	// check for "all" arg
	if (!str_cmp(abil_arg, "all")) {
		all = TRUE;
	}
	
	if (is_abbrev(cmd_arg, "add")) {
		if (!*abil_arg) {
			msg_to_char(ch, "Add what ability?\r\n");
		}
		else if (!(abil = find_ability(abil_arg))) {
			msg_to_char(ch, "Unknown ability '%s'.\r\n", abil_arg);
		}
		else if (is_class_ability(abil)) {
			msg_to_char(ch, "You can't assign %s to this skill because it's already assigned to a class.\r\n", ABIL_NAME(abil));
		}
		else if (ABIL_ASSIGNED_SKILL(abil) && SKILL_VNUM(ABIL_ASSIGNED_SKILL(abil)) != SKILL_VNUM(skill)) {
			msg_to_char(ch, "You can't assign %s to this skill because it's already assigned to [%d] %s.\r\n", ABIL_NAME(abil), SKILL_VNUM(ABIL_ASSIGNED_SKILL(abil)), SKILL_NAME(ABIL_ASSIGNED_SKILL(abil)));
		}
		else if (ABIL_IS_SYNERGY(abil)) {
			msg_to_char(ch, "You can't assign %s to this skill because it is a synergy ability.\r\n", ABIL_NAME(abil));
		}
		else if (has_ability_data_any(abil, ADL_PARENT)) {
			msg_to_char(ch, "You can't assign %s to this skill because it has PARENT data.\r\n", ABIL_NAME(abil));
		}
		else if (!*sub_arg || !isdigit(*sub_arg) || (level = atoi(sub_arg)) < 0) {
			msg_to_char(ch, "Add the ability at what level?\r\n");
		}
		else if (*req_arg && str_cmp(req_arg, "none") && !(requires = find_ability(req_arg))) {
			msg_to_char(ch, "Invalid pre-requisite ability '%s'.\r\n", req_arg);
		}
		else if (abil == requires) {
			msg_to_char(ch, "It cannot require itself.\r\n");
		}
		else {
			// this does some validation
			found = found_prq = FALSE;
			change = NULL;
			LL_FOREACH(SKILL_ABILITIES(skill), skab) {
				if (skab->vnum == ABIL_VNUM(abil)) {
					change = skab;
					found = TRUE;
				}
				if (requires && skab->vnum == ABIL_VNUM(requires)) {
					found_prq = TRUE;
				}
			}
			
			if (requires && !found_prq) {
				msg_to_char(ch, "You can't add a prerequisite that isn't assigned to this skill.\r\n");
				return;
			}
			
			if (found && change) {
				change->level = level;
				change->prerequisite = requires ? ABIL_VNUM(requires) : NOTHING;
			}
			else {
				CREATE(skab, struct skill_ability, 1);
				skab->vnum = ABIL_VNUM(abil);
				skab->level = level;
				skab->prerequisite = requires ? ABIL_VNUM(requires) : NOTHING;
				LL_APPEND(SKILL_ABILITIES(skill), skab);
			}
			
			msg_to_char(ch, "You assign %s at level %d", ABIL_NAME(abil), level);
			if (requires) {
				msg_to_char(ch, " (branching from %s).\r\n", ABIL_NAME(requires));
			}
			else {
				msg_to_char(ch, ".\r\n");
			}
			
			// in case
			LL_SORT(SKILL_ABILITIES(skill), sort_skill_abilities);
		}
	}
	else if (is_abbrev(cmd_arg, "remove")) {
		if (!all && !*abil_arg) {
			msg_to_char(ch, "Remove which ability?\r\n");
			return;
		}
		if (!all && !(abil = find_ability_on_skill(abil_arg, skill))) {
			msg_to_char(ch, "There is no such ability on this skill.\r\n");
			return;
		}
		
		found = found_prq = FALSE;
		LL_FOREACH_SAFE(SKILL_ABILITIES(skill), skab, next_skab) {
			if (all || (abil && skab->vnum == ABIL_VNUM(abil))) {
				LL_DELETE(SKILL_ABILITIES(skill), skab);
				free(skab);
				found = TRUE;
			}
			else if (abil && skab->prerequisite == ABIL_VNUM(abil)) {
				skab->prerequisite = NO_ABIL;
				found = found_prq = TRUE;
			}
		}
		
		if (!found) {
			msg_to_char(ch, "Couldn't find anything to remove.\r\n");
		}
		else if (all) {
			msg_to_char(ch, "You remove all abilities.\r\n");
		}
		else {
			msg_to_char(ch, "You remove the %s ability%s.\r\n", ABIL_NAME(abil), found_prq ? " (and things that required it no longer require anything)" : "");
		}
	}
	else if (is_abbrev(cmd_arg, "change")) {
		if (!*abil_arg) {
			msg_to_char(ch, "Change which ability?\r\n");
		}
		else if (!(abil = find_ability_on_skill(abil_arg, skill))) {
			msg_to_char(ch, "There is no such ability on this skill.\r\n");
		}
		else if (is_abbrev(sub_arg, "level")) {
			if (!*req_arg || !isdigit(*req_arg) || (level = atoi(req_arg)) < 0) {
				msg_to_char(ch, "Set it to what level?\r\n");
				return;
			}
			
			found = FALSE;
			LL_FOREACH(SKILL_ABILITIES(skill), skab) {
				if (skab->vnum == ABIL_VNUM(abil)) {
					skab->level = level;
					found = TRUE;
				}
			}
			
			if (found) {
				msg_to_char(ch, "You change %s to level %d.\r\n", ABIL_NAME(abil), level);
			}
			else {
				msg_to_char(ch, "%s is not assigned to this skill.\r\n", ABIL_NAME(abil));
			}
		}
		else if (is_abbrev(sub_arg, "requires") || is_abbrev(sub_arg, "requirement") || is_abbrev(sub_arg, "prerequisite")) {
			if (!*req_arg) {
				msg_to_char(ch, "Require which ability (or none)?\r\n");
			}
			else if (str_cmp(req_arg, "none") && !(requires = find_ability(req_arg))) {
				msg_to_char(ch, "Invalid pre-requisite ability '%s'.\r\n", req_arg);
			}
			else {
				found = FALSE;
				found_prq = (requires == NULL);
				change = NULL;
				LL_FOREACH(SKILL_ABILITIES(skill), skab) {
					if (skab->vnum == ABIL_VNUM(abil)) {
						change = skab;
						found = TRUE;
					}
					else if (requires && skab->vnum == ABIL_VNUM(requires)) {
						found_prq = TRUE;
					}
				}
				
				if (!found || !change) {
					msg_to_char(ch, "%s is not assigned to this skill.\r\n", ABIL_NAME(abil));
				}
				else if (!found_prq) {
					msg_to_char(ch, "You can't require an ability that isn't assigned to this skill.\r\n");
				}
				else {
					change->prerequisite = requires ? ABIL_VNUM(requires) : -1;
					if (requires) {
						msg_to_char(ch, "%s now requires the %s ability.\r\n", ABIL_NAME(abil), ABIL_NAME(requires));
					}
					else {
						msg_to_char(ch, "%s no longer requires a prerequisite.\r\n", ABIL_NAME(abil));
					}
				}
			}
		}
		else {
			msg_to_char(ch, "You can change the level or requirement.\r\n");
		}
		
		// in case
		LL_SORT(SKILL_ABILITIES(skill), sort_skill_abilities);
	}
	else {
		msg_to_char(ch, "Usage: tree add <ability> <level> [requires ability]\r\n");
		msg_to_char(ch, "       tree remove <ability | all>\r\n");
		msg_to_char(ch, "       tree change <ability> <level | requires> <new value>\r\n");
	}
}
