/* ************************************************************************
*   File: olc.trigger.c                                   EmpireMUD 2.0b5 *
*  Usage: OLC for triggers                                                *
*                                                                         *
*  EmpireMUD code base by Paul Clarke, (C) 2000-2024                      *
*  All rights reserved.  See license.doc for complete information.        *
*                                                                         *
*  EmpireMUD based upon CircleMUD 3.0, bpl 17, by Jeremy Elson.           *
*  CircleMUD (C) 1993, 94 by the Trustees of the Johns Hopkins University *
*  CircleMUD is based on DikuMUD, Copyright (C) 1990, 1991.               *
************************************************************************ */
#include "conf.h"
#include "sysdep.h"

#include "structs.h"
#include "utils.h"
#include "interpreter.h"
#include "db.h"
#include "comm.h"
#include "olc.h"
#include "handler.h"
#include "dg_scripts.h"
#include "dg_event.h"
#include "constants.h"

/**
* Contents:
*   Helpers
*   Displays
*   Edit Modules
*/

// external functions
EVENT_CANCEL_FUNC(cancel_wait_event);
struct cmdlist_element *compile_command_list(char *input);


// locals
const char *trig_arg_obj_where[] = {
	"equipment",
	"inventory",
	"room",
	"\n"
};

const char *trig_arg_phrase_type[] = {
	"phrase",
	"wordlist",
	"\n"
};

const char *default_trig_name = "new trigger";


 //////////////////////////////////////////////////////////////////////////////
//// HELPERS /////////////////////////////////////////////////////////////////

/**
* Checks for common trigger problems and reports them to ch.
*
* @param trig_data *trigger The thing to audit.
* @param char_data *ch The person to report to.
* @return bool TRUE if any problems were reported; FALSE if all good.
*/
bool audit_trigger(trig_data *trig, char_data *ch) {
	struct cmdlist_element *cmd;
	bool problem = FALSE;
	bitvector_t bits;
	int pos;
	bool links;
	struct trig_link *link;
		
	if (!str_cmp(GET_TRIG_NAME(trig), default_trig_name)) {
		olc_audit_msg(ch, GET_TRIG_VNUM(trig), "Name not set");
		problem = TRUE;
	}
	
	// look for a 'percent' type with a 0%
	for (pos = 0, bits = GET_TRIG_TYPE(trig); bits; ++pos, bits >>= 1) {
		if (IS_SET(bits, BIT(0)) && trig_argument_type_list[trig->attach_type][pos] == TRIG_ARG_PERCENT && GET_TRIG_NARG(trig) == 0) {
			olc_audit_msg(ch, GET_TRIG_VNUM(trig), "Trigger has 0%% chance");
			problem = TRUE;
		}
	}
	
	links = FALSE;
	
	LL_FOREACH(trig->cmdlist, cmd) {
		if (strlen(cmd->cmd) > 255) {
			olc_audit_msg(ch, GET_TRIG_VNUM(trig), "Trigger has a line longer than 255 characters and won't save properly");
			problem = TRUE;
		}
		
		// check for links
		for (pos = 0; pos < strlen(cmd->cmd) - 1 && !links; ++pos) {
			if (isdigit(cmd->cmd[pos]) && isdigit(cmd->cmd[pos+1])) {
				links = TRUE;
			}
		}
	}
	
	if (links && !GET_TRIG_LINKS(trig)) {
		olc_audit_msg(ch, GET_TRIG_VNUM(trig), "Contains numbers but has no links");
		problem = TRUE;
	}
	
	LL_FOREACH(GET_TRIG_LINKS(trig), link) {
		if (!str_cmp(get_name_by_olc_type(link->type, link->vnum), "Unknown")) {
			olc_audit_msg(ch, GET_TRIG_VNUM(trig), "Contains link to Unknown/missing thing");
			problem = TRUE;
		}
	}
	
	return problem;
}


/**
* Determines all the argument types to show for the various types of a trigger.
*
* @param trig_data *trig The trigger.
* @return bitvector_t A set of TRIG_ARG_x flags.
*/
bitvector_t compile_argument_types_for_trigger(trig_data *trig) {
	bitvector_t flags = NOBITS, trigger_types = GET_TRIG_TYPE(trig);
	int pos;
	
	for (pos = 0; trigger_types; trigger_types >>= 1, ++pos) {
		if (IS_SET(trigger_types, BIT(0))) {
			flags |= trig_argument_type_list[trig->attach_type][pos];
		}
	}
	
	return flags;
}


/**
* Creates a new trigger entry.
* 
* @param trig_vnum vnum The number to create.
* @return trig_data* The new trigger.
*/
trig_data *create_trigger_table_entry(trig_vnum vnum) {
	trig_data *trig;
	
	// sanity
	if (real_trigger(vnum)) {
		log("SYSERR: Attempting to insert trigger at existing vnum %d", vnum);
		return real_trigger(vnum);
	}
	
	CREATE(trig, trig_data, 1);
	trig_data_init(trig);
	trig->vnum = vnum;
	add_trigger_to_table(trig);
	
	// simple default data (triggers cannot be nameless)
	trig->name = str_dup(default_trig_name);
		
	// save index and crop file now
	save_index(DB_BOOT_TRG);
	save_library_file_for_vnum(DB_BOOT_TRG, vnum);
	
	return trig;
}


/**
* For the .list command.
*
* @param trig_data *trig The thing to list.
* @param bool detail If TRUE, provide additional details
* @return char* The line to show (without a CRLF).
*/
char *list_one_trigger(trig_data *trig, bool detail) {
	static char output[MAX_STRING_LENGTH];
	
	if (detail) {
		safe_snprintf(output, sizeof(output), "[%5d] %s", GET_TRIG_VNUM(trig), GET_TRIG_NAME(trig));
	}
	else {
		safe_snprintf(output, sizeof(output), "[%5d] %s", GET_TRIG_VNUM(trig), GET_TRIG_NAME(trig));
	}
	
	return output;
}


/**
* Removes triggers from a live set of scripts, by vnum.
*
* @param struct script_data *script The script to remove triggers from.
* @param trig_vnum vnum The trigger vnum to remove.
* @return bool TRUE if any were removed; FALSE otherwise.
*/
bool remove_live_script_by_vnum(struct script_data *script, trig_vnum vnum) {
	trig_data *trig, *next_trig;
	bool found = FALSE;
	
	if (!script) {
		return found;
	}

	for (trig = TRIGGERS(script); trig; trig = next_trig) {
		next_trig = trig->next;
		
		if (GET_TRIG_VNUM(trig) == vnum) {
			found = TRUE;
			LL_DELETE(TRIGGERS(script), trig);
			extract_trigger(trig);
		}
	}
	
	return found;
}


/**
* Ensures there are no non-existent triggers attached to anything.
*/
void check_triggers(void) {
	struct trig_proto_list *tpl, *next_tpl;
	quest_data *quest, *next_quest;
	room_template *rmt, *next_rmt;
	vehicle_data *veh, *next_veh;
	char_data *mob, *next_mob;
	obj_data *obj, *next_obj;
	bld_data *bld, *next_bld;
	
	// check building protos
	HASH_ITER(hh, building_table, bld, next_bld) {
		LL_FOREACH_SAFE(GET_BLD_SCRIPTS(bld), tpl, next_tpl) {
			if (!real_trigger(tpl->vnum)) {
				log("SYSERR: Removing missing trigger %d from building %d.", tpl->vnum, GET_BLD_VNUM(bld));
				LL_DELETE(GET_BLD_SCRIPTS(bld), tpl);
				free(tpl);
			}
		}
	}
	
	// check mob protos
	HASH_ITER(hh, mobile_table, mob, next_mob) {
		LL_FOREACH_SAFE(mob->proto_script, tpl, next_tpl) {
			if (!real_trigger(tpl->vnum)) {
				log("SYSERR: Removing missing trigger %d from mobile %d.", tpl->vnum, GET_MOB_VNUM(mob));
				LL_DELETE(mob->proto_script, tpl);
				free(tpl);
			}
		}
	}
	
	// check obj protos
	HASH_ITER(hh, object_table, obj, next_obj) {
		LL_FOREACH_SAFE(obj->proto_script, tpl, next_tpl) {
			if (!real_trigger(tpl->vnum)) {
				log("SYSERR: Removing missing trigger %d from object %d.", tpl->vnum, GET_OBJ_VNUM(obj));
				LL_DELETE(obj->proto_script, tpl);
				free(tpl);
			}
		}
	}
	
	// check quests
	HASH_ITER(hh, quest_table, quest, next_quest) {
		LL_FOREACH_SAFE(QUEST_SCRIPTS(quest), tpl, next_tpl) {
			if (!real_trigger(tpl->vnum)) {
				log("SYSERR: Removing missing trigger %d from quest %d.", tpl->vnum, QUEST_VNUM(quest));
				LL_DELETE(QUEST_SCRIPTS(quest), tpl);
				free(tpl);
			}
		}
	}
	
	// check room templates
	HASH_ITER(hh, room_template_table, rmt, next_rmt) {
		LL_FOREACH_SAFE(GET_RMT_SCRIPTS(rmt), tpl, next_tpl) {
			if (!real_trigger(tpl->vnum)) {
				log("SYSERR: Removing missing trigger %d from room template %d.", tpl->vnum, GET_RMT_VNUM(rmt));
				LL_DELETE(GET_RMT_SCRIPTS(rmt), tpl);
				free(tpl);
			}
		}
	}
	
	// check vehicle protos
	HASH_ITER(hh, vehicle_table, veh, next_veh) {
		LL_FOREACH_SAFE(veh->proto_script, tpl, next_tpl) {
			if (!real_trigger(tpl->vnum)) {
				log("SYSERR: Removing missing trigger %d from vehicle %d.", tpl->vnum, VEH_VNUM(veh));
				LL_DELETE(veh->proto_script, tpl);
				free(tpl);
			}
		}
	}
}


/**
* Deletes a trigger from a proto list.
*
* @param struct trig_proto_list **list The list to check.
* @param trig_vnum vnum The trigger to remove.
* @return bool TRUE if any triggers were removed, FALSE if not.
*/
bool delete_from_proto_list_by_vnum(struct trig_proto_list **list, trig_vnum vnum) {
	struct trig_proto_list *trig, *next_trig;
	bool found = FALSE;
	
	for (trig = *list; trig; trig = next_trig) {
		next_trig = trig->next;
		if (trig->vnum == vnum) {
			found = TRUE;
			LL_DELETE(*list, trig);
			free(trig);
		}
	}
	
	return found;
}


/**
* WARNING: This function actually deletes a trigger.
*
* @param char_data *ch The person doing the deleting.
* @param trig_vnum vnum The vnum to delete.
*/
void olc_delete_trigger(char_data *ch, trig_vnum vnum) {
	trig_data *trig, *trig_iter, *next_trig;
	quest_data *quest, *next_quest;
	room_template *rmt, *next_rmt;
	vehicle_data *veh, *next_veh;
	room_data *room, *next_room;
	char_data *mob, *next_mob;
	shop_data *shop, *next_shop;
	descriptor_data *dsc;
	adv_data *adv, *next_adv;
	obj_data *obj, *next_obj;
	bld_data *bld, *next_bld;
	char name[256];
	bool found;

	if (!(trig = real_trigger(vnum))) {
		msg_to_char(ch, "There is no such trigger %d.\r\n", vnum);
		return;
	}
	
	safe_snprintf(name, sizeof(name), "%s", NULLSAFE(GET_TRIG_NAME(trig)));
	
	if (HASH_COUNT(trigger_table) <= 1) {
		msg_to_char(ch, "You can't delete the last trigger.\r\n");
		return;
	}
	
	// look for live mobs with this script and remove
	DL_FOREACH(character_list, mob) {
		if (IS_NPC(mob) && SCRIPT(mob)) {
			remove_live_script_by_vnum(SCRIPT(mob), vnum);
			check_extract_script(mob, MOB_TRIGGER);
		}
	}
	
	// look for live objs with this script and remove
	DL_FOREACH(object_list, obj) {
		if (SCRIPT(obj)) {
			remove_live_script_by_vnum(SCRIPT(obj), vnum);
			check_extract_script(obj, OBJ_TRIGGER);
		}
	}
	
	// live vehicles -> remove
	DL_FOREACH(vehicle_list, veh) {
		if (SCRIPT(veh)) {
			remove_live_script_by_vnum(SCRIPT(veh), vnum);
			check_extract_script(veh, VEH_TRIGGER);
		}
	}
	
	// look for live rooms with this trigger
	HASH_ITER(hh, world_table, room, next_room) {
		if (SCRIPT(room)) {
			remove_live_script_by_vnum(SCRIPT(room), vnum);
			check_extract_script(room, WLD_TRIGGER);
		}
		delete_from_proto_list_by_vnum(&(room->proto_script), vnum);
	}
	
	// free them before continuing (or risk memory error doom)
	free_freeable_triggers();
	
	// remove from hash table (AFTER deleting live copies)
	remove_trigger_from_table(trig);
	
	// remove from adventures
	HASH_ITER(hh, adventure_table, adv, next_adv) {
		if (delete_from_proto_list_by_vnum(&GET_ADV_SCRIPTS(adv), vnum)) {
			syslog(SYS_OLC, GET_INVIS_LEV(ch), TRUE, "OLC: Adventure %d %s lost deleted trigger", GET_ADV_VNUM(adv), GET_ADV_NAME(adv));
			save_library_file_for_vnum(DB_BOOT_ADV, GET_ADV_VNUM(adv));
		}
	}
	
	// update building protos
	HASH_ITER(hh, building_table, bld, next_bld) {
		if (delete_from_proto_list_by_vnum(&GET_BLD_SCRIPTS(bld), vnum)) {
			syslog(SYS_OLC, GET_INVIS_LEV(ch), TRUE, "OLC: Building %d %s lost deleted trigger", GET_BLD_VNUM(bld), GET_BLD_NAME(bld));
			save_library_file_for_vnum(DB_BOOT_BLD, GET_BLD_VNUM(bld));
		}
	}
	
	// update mob protos
	HASH_ITER(hh, mobile_table, mob, next_mob) {
		if (delete_from_proto_list_by_vnum(&mob->proto_script, vnum)) {
			syslog(SYS_OLC, GET_INVIS_LEV(ch), TRUE, "OLC: Mobile %d %s lost deleted trigger", GET_MOB_VNUM(mob), GET_SHORT_DESC(mob));
			save_library_file_for_vnum(DB_BOOT_MOB, mob->vnum);
		}
	}
	
	// update obj protos
	HASH_ITER(hh, object_table, obj, next_obj) {
		if (delete_from_proto_list_by_vnum(&obj->proto_script, vnum)) {
			syslog(SYS_OLC, GET_INVIS_LEV(ch), TRUE, "OLC: Object %d %s lost deleted trigger", GET_OBJ_VNUM(obj), GET_OBJ_SHORT_DESC(obj));
			save_library_file_for_vnum(DB_BOOT_OBJ, GET_OBJ_VNUM(obj));
		}
	}
	
	// update quests
	HASH_ITER(hh, quest_table, quest, next_quest) {
		found = delete_quest_giver_from_list(&QUEST_STARTS_AT(quest), QG_TRIGGER, vnum);
		found |= delete_quest_giver_from_list(&QUEST_ENDS_AT(quest), QG_TRIGGER, vnum);
		found |= delete_from_proto_list_by_vnum(&QUEST_SCRIPTS(quest), vnum);
		
		if (found) {
			SET_BIT(QUEST_FLAGS(quest), QST_IN_DEVELOPMENT);
			syslog(SYS_OLC, GET_INVIS_LEV(ch), TRUE, "OLC: Quest %d %s set IN-DEV due to deleted trigger", QUEST_VNUM(quest), QUEST_NAME(quest));
			save_library_file_for_vnum(DB_BOOT_QST, QUEST_VNUM(quest));
		}
	}
	
	// room templates
	HASH_ITER(hh, room_template_table, rmt, next_rmt) {
		if (delete_from_proto_list_by_vnum(&GET_RMT_SCRIPTS(rmt), vnum)) {
			syslog(SYS_OLC, GET_INVIS_LEV(ch), TRUE, "OLC: Room template %d %s lost deleted trigger", GET_RMT_VNUM(rmt), GET_RMT_TITLE(rmt));
			save_library_file_for_vnum(DB_BOOT_RMT, GET_RMT_VNUM(rmt));
		}
	}
	
	// update shops
	HASH_ITER(hh, shop_table, shop, next_shop) {
		found = delete_quest_giver_from_list(&SHOP_LOCATIONS(shop), QG_TRIGGER, vnum);
		
		if (found) {
			SET_BIT(SHOP_FLAGS(shop), SHOP_IN_DEVELOPMENT);
			syslog(SYS_OLC, GET_INVIS_LEV(ch), TRUE, "OLC: Shop %d %s set IN-DEV due to deleted trigger", SHOP_VNUM(shop), SHOP_NAME(shop));
			save_library_file_for_vnum(DB_BOOT_SHOP, SHOP_VNUM(shop));
		}
	}
	
	// update triggers
	HASH_ITER(hh, trigger_table, trig_iter, next_trig) {
		found = trigger_has_link(trig_iter, OLC_TRIGGER, vnum);
		if (found) {
			syslog(SYS_OLC, GET_INVIS_LEV(ch), TRUE, "OLC: Trigger %d %s lost link to trigger [%d] %s", GET_TRIG_VNUM(trig_iter), GET_TRIG_NAME(trig_iter), vnum, name);
			// Doesn't delete
			// save_library_file_for_vnum(DB_BOOT_TRG, GET_TRIG_VNUM(trig));
		}
	}
	
	// update vehicle protos
	HASH_ITER(hh, vehicle_table, veh, next_veh) {
		if (delete_from_proto_list_by_vnum(&veh->proto_script, vnum)) {
			syslog(SYS_OLC, GET_INVIS_LEV(ch), TRUE, "OLC: Vehicle %d %s lost deleted trigger", VEH_VNUM(veh), VEH_SHORT_DESC(veh));
			save_library_file_for_vnum(DB_BOOT_VEH, VEH_VNUM(veh));
		}
	}
	
	// update olc editors
	for (dsc = descriptor_list; dsc; dsc = dsc->next) {
		if (GET_OLC_ADVENTURE(dsc) && delete_from_proto_list_by_vnum(&GET_ADV_SCRIPTS(GET_OLC_ADVENTURE(dsc)), vnum)) {
			msg_to_char(dsc->character, "A trigger attached to the adventure you're editing was deleted.\r\n");
		}
		if (GET_OLC_BUILDING(dsc) && delete_from_proto_list_by_vnum(&GET_BLD_SCRIPTS(GET_OLC_BUILDING(dsc)), vnum)) {
			msg_to_char(dsc->character, "A trigger attached to the building you're editing was deleted.\r\n");
		}
		if (GET_OLC_OBJECT(dsc) && delete_from_proto_list_by_vnum(&GET_OLC_OBJECT(dsc)->proto_script, vnum)) {
			msg_to_char(dsc->character, "A trigger attached to the object you're editing was deleted.\r\n");
		}
		if (GET_OLC_MOBILE(dsc) && delete_from_proto_list_by_vnum(&GET_OLC_MOBILE(dsc)->proto_script, vnum)) {
			msg_to_char(dsc->character, "A trigger attached to the mobile you're editing was deleted.\r\n");
		}
		if (GET_OLC_QUEST(dsc)) {
			found = delete_quest_giver_from_list(&QUEST_STARTS_AT(GET_OLC_QUEST(dsc)), QG_TRIGGER, vnum);
			found |= delete_quest_giver_from_list(&QUEST_ENDS_AT(GET_OLC_QUEST(dsc)), QG_TRIGGER, vnum);
			found |= delete_from_proto_list_by_vnum(&QUEST_SCRIPTS(GET_OLC_QUEST(dsc)), vnum);
			
			if (found) {
				SET_BIT(QUEST_FLAGS(GET_OLC_QUEST(dsc)), QST_IN_DEVELOPMENT);
				msg_to_desc(dsc, "A trigger used by the quest you are editing was deleted.\r\n");
			}
		}
		if (GET_OLC_SHOP(dsc)) {
			found = delete_quest_giver_from_list(&SHOP_LOCATIONS(GET_OLC_SHOP(dsc)), QG_TRIGGER, vnum);
			
			if (found) {
				SET_BIT(SHOP_FLAGS(GET_OLC_SHOP(dsc)), SHOP_IN_DEVELOPMENT);
				msg_to_desc(dsc, "A trigger used by the shop you are editing was deleted.\r\n");
			}
		}
		if (GET_OLC_TRIGGER(dsc)) {
			found = trigger_has_link(GET_OLC_TRIGGER(dsc), OLC_TRIGGER, vnum);
			if (found) {
				msg_to_desc(dsc, "Trigger [%d] %s was deleted but remains in the link list for the trigger you're editing.", vnum, name);
			}
		}
		if (GET_OLC_ROOM_TEMPLATE(dsc) && delete_from_proto_list_by_vnum(&GET_OLC_ROOM_TEMPLATE(dsc)->proto_script, vnum)) {
			msg_to_char(dsc->character, "A trigger attached to the room template you're editing was deleted.\r\n");
		}
		if (GET_OLC_VEHICLE(dsc) && delete_from_proto_list_by_vnum(&GET_OLC_VEHICLE(dsc)->proto_script, vnum)) {
			msg_to_char(dsc->character, "A trigger attached to the vehicle you're editing was deleted.\r\n");
		}
	}
	
	// save index and trigger file now
	save_index(DB_BOOT_TRG);
	save_library_file_for_vnum(DB_BOOT_TRG, vnum);
	
	syslog(SYS_OLC, GET_INVIS_LEV(ch), TRUE, "OLC: %s has deleted trigger %d %s", GET_NAME(ch), vnum, name);
	msg_to_char(ch, "Trigger %d (%s) deleted.\r\n", vnum, name);
	
	free_trigger(trig);
}


/**
* Searches properties of triggers.
*
* @param char_data *ch The person searching.
* @param char *argument The argument they entered.
*/
void olc_fullsearch_trigger(char_data *ch, char *argument) {
	char type_arg[MAX_INPUT_LENGTH], val_arg[MAX_INPUT_LENGTH], find_keywords[MAX_INPUT_LENGTH];
	int count, lookup, only_attaches = NOTHING, vmin = NOTHING, vmax = NOTHING;
	bitvector_t mob_types = NOBITS, obj_types = NOBITS, wld_types = NOBITS, veh_types = NOBITS;
	trig_data *trig, *next_trig;
	struct cmdlist_element *cmd;
	bool any_types = FALSE, any;
	
	if (!*argument) {
		msg_to_char(ch, "See HELP TEDIT FULLSEARCH for syntax.\r\n");
		return;
	}
	
	// process argument
	*find_keywords = '\0';
	while (*argument) {
		// figure out a type
		argument = any_one_arg(argument, type_arg);
		
		if (!strcmp(type_arg, "-")) {
			continue;	// just skip stray dashes
		}
		
		FULLSEARCH_LIST("attaches", only_attaches, trig_attach_types)
		FULLSEARCH_INT("vmin", vmin, 0, INT_MAX)
		FULLSEARCH_INT("vmax", vmax, 0, INT_MAX)
		
		// custom:
		else if (is_abbrev(type_arg, "-types")) {
			argument = any_one_word(argument, val_arg);
			any = FALSE;
			if ((lookup = search_block(val_arg, trig_types, FALSE)) != NOTHING) {
				mob_types |= BIT(lookup);
				any_types = any = TRUE;
			}
			if ((lookup = search_block(val_arg, otrig_types, FALSE)) != NOTHING) {
				obj_types |= BIT(lookup);
				any_types = any = TRUE;
			}
			if ((lookup = search_block(val_arg, vtrig_types, FALSE)) != NOTHING) {
				veh_types |= BIT(lookup);
				any_types = any = TRUE;
			}
			if ((lookup = search_block(val_arg, wtrig_types, FALSE)) != NOTHING) {
				wld_types |= BIT(lookup);
				any_types = any = TRUE;
			}
			
			if (!any) {
				msg_to_char(ch, "Invalid trigger type '%s'.\r\n", val_arg);
				return;
			}
		}
		else {	// not sure what to do with it? treat it like a keyword
			sprintf(find_keywords + strlen(find_keywords), "%s%s", *find_keywords ? " " : "", type_arg);
		}
		
		// prepare for next loop
		skip_spaces(&argument);
	}
	
	build_page_display(ch, "Trigger fullsearch: %s", show_color_codes(find_keywords));
	count = 0;
	
	// okay now look up items
	HASH_ITER(hh, trigger_table, trig, next_trig) {
		if ((vmin != NOTHING && GET_TRIG_VNUM(trig) < vmin) || (vmax != NOTHING && GET_TRIG_VNUM(trig) > vmax)) {
			continue;	// vnum range
		}
		if (only_attaches != NOTHING && trig->attach_type != only_attaches) {
			continue;
		}
		// x_TRIGGER:
		if (mob_types && trig->attach_type == MOB_TRIGGER && (GET_TRIG_TYPE(trig) & mob_types) != mob_types) {
			continue;
		}
		if (obj_types && trig->attach_type == OBJ_TRIGGER && (GET_TRIG_TYPE(trig) & obj_types) != obj_types) {
			continue;
		}
		if (wld_types && (trig->attach_type == WLD_TRIGGER || trig->attach_type == ADV_TRIGGER || trig->attach_type == RMT_TRIGGER || trig->attach_type == BLD_TRIGGER) && (GET_TRIG_TYPE(trig) & wld_types) != wld_types) {
			continue;
		}
		if (veh_types && trig->attach_type == VEH_TRIGGER && (GET_TRIG_TYPE(trig) & veh_types) != veh_types) {
			continue;
		}
		if (any_types && trig->attach_type == MOB_TRIGGER && !mob_types) {
			continue;
		}
		if (any_types && trig->attach_type == OBJ_TRIGGER && !obj_types) {
			continue;
		}
		if (any_types && (trig->attach_type == WLD_TRIGGER || trig->attach_type == ADV_TRIGGER || trig->attach_type == RMT_TRIGGER || trig->attach_type == BLD_TRIGGER) && !wld_types) {
			continue;
		}
		if (any_types && trig->attach_type == VEH_TRIGGER && !veh_types) {
			continue;
		}
		if (*find_keywords) {
			any = multi_isname(find_keywords, GET_TRIG_NAME(trig));	// check name first
			if (!any && GET_TRIG_ARG(trig)) {
				any |= multi_isname(find_keywords, GET_TRIG_ARG(trig));	// text arg
			}
			
			if (!any) {
				LL_FOREACH(trig->cmdlist, cmd) {
					if (strstr(cmd->cmd, find_keywords)) {
						any = TRUE;
						break;
					}
				}
			}
			if (!any) {
				continue;
			}
		}
		
		// show it
		build_page_display(ch, "[%5d] %s", GET_TRIG_VNUM(trig), GET_TRIG_NAME(trig));
		++count;
	}
	
	if (count > 0) {
		build_page_display(ch, "(%d triggers)", count);
	}
	else {
		build_page_display_str(ch, " none");
	}
	
	send_page_display(ch);
}


/**
* Searches for all uses of a crop and displays them.
*
* @param char_data *ch The player.
* @param crop_vnum vnum The crop vnum.
*/
void olc_search_trigger(char_data *ch, trig_vnum vnum) {
	trig_data *proto = real_trigger(vnum);
	quest_data *quest, *next_quest;
	struct trig_proto_list *trig;
	room_template *rmt, *next_rmt;
	vehicle_data *veh, *next_veh;
	shop_data *shop, *next_shop;
	char_data *mob, *next_mob;
	adv_data *adv, *next_adv;
	obj_data *obj, *next_obj;
	bld_data *bld, *next_bld;
	trig_data *trig_iter, *next_trig;
	int found;
	bool any;
	
	if (!proto) {
		msg_to_char(ch, "There is no trigger %d.\r\n", vnum);
		return;
	}
	
	found = 0;
	build_page_display(ch, "Occurrences of trigger %d (%s):", vnum, GET_TRIG_NAME(proto));
	
	// adventure scripts
	HASH_ITER(hh, adventure_table, adv, next_adv) {
		any = FALSE;
		for (trig = GET_ADV_SCRIPTS(adv); trig && !any; trig = trig->next) {
			if (trig->vnum == vnum) {
				any = TRUE;
				++found;
				build_page_display(ch, "ADV [%5d] %s", GET_ADV_VNUM(adv), GET_ADV_NAME(adv));
			}
		}
	}
	
	// buildings
	HASH_ITER(hh, building_table, bld, next_bld) {
		any = FALSE;
		for (trig = GET_BLD_SCRIPTS(bld); trig && !any; trig = trig->next) {
			if (trig->vnum == vnum) {
				any = TRUE;
				++found;
				build_page_display(ch, "BLD [%5d] %s", GET_BLD_VNUM(bld), GET_BLD_NAME(bld));
			}
		}
	}
	
	// mobs
	HASH_ITER(hh, mobile_table, mob, next_mob) {
		any = FALSE;
		for (trig = mob->proto_script; trig && !any; trig = trig->next) {
			if (trig->vnum == vnum) {
				any = TRUE;
				++found;
				build_page_display(ch, "MOB [%5d] %s", GET_MOB_VNUM(mob), GET_SHORT_DESC(mob));
			}
		}
	}
	
	// objs
	HASH_ITER(hh, object_table, obj, next_obj) {
		any = FALSE;
		for (trig = obj->proto_script; trig && !any; trig = trig->next) {
			if (trig->vnum == vnum) {
				any = TRUE;
				++found;
				build_page_display(ch, "OBJ [%5d] %s", GET_OBJ_VNUM(obj), GET_OBJ_SHORT_DESC(obj));
			}
		}
	}
	
	// quests
	HASH_ITER(hh, quest_table, quest, next_quest) {
		any = find_quest_giver_in_list(QUEST_STARTS_AT(quest), QG_TRIGGER, vnum);
		any |= find_quest_giver_in_list(QUEST_ENDS_AT(quest), QG_TRIGGER, vnum);
		if (!any) {
			LL_FOREACH(QUEST_SCRIPTS(quest), trig) {
				if (trig->vnum == vnum) {
					any = TRUE;
					break;
				}
			}
		}
		
		if (any) {
			++found;
			build_page_display(ch, "QST [%5d] %s", QUEST_VNUM(quest), QUEST_NAME(quest));
		}
	}
	
	// room templates
	HASH_ITER(hh, room_template_table, rmt, next_rmt) {
		any = FALSE;
		for (trig = GET_RMT_SCRIPTS(rmt); trig && !any; trig = trig->next) {
			if (trig->vnum == vnum) {
				any = TRUE;
				++found;
				build_page_display(ch, "RMT [%5d] %s", GET_RMT_VNUM(rmt), GET_RMT_TITLE(rmt));
			}
		}
	}
	
	// shops
	HASH_ITER(hh, shop_table, shop, next_shop) {
		any = find_quest_giver_in_list(SHOP_LOCATIONS(shop), QG_TRIGGER, vnum);
		
		if (any) {
			++found;
			build_page_display(ch, "SHOP [%5d] %s", SHOP_VNUM(shop), SHOP_NAME(shop));
		}
	}
	
	// triggers
	HASH_ITER(hh, trigger_table, trig_iter, next_trig) {
		if (trigger_has_link(trig_iter, OLC_TRIGGER, vnum)) {
			++found;
			build_page_display(ch, "TRG [%5d] %s", GET_TRIG_VNUM(trig_iter), GET_TRIG_NAME(trig_iter));
		}
	}
	
	// vehicles
	HASH_ITER(hh, vehicle_table, veh, next_veh) {
		any = FALSE;
		for (trig = veh->proto_script; trig && !any; trig = trig->next) {
			if (trig->vnum == vnum) {
				any = TRUE;
				++found;
				build_page_display(ch, "VEH [%5d] %s", VEH_VNUM(veh), VEH_SHORT_DESC(veh));
			}
		}
	}
	
	if (found > 0) {
		build_page_display(ch, "%d location%s shown", found, PLURAL(found));
	}
	else {
		build_page_display_str(ch, " none");
	}
	
	send_page_display(ch);
}


/**
* Function to save a player's changes to a trigger (or a new one).
*
* @param descriptor_data *desc The descriptor who is saving.
*/
void save_olc_trigger(descriptor_data *desc, char *script_text) {
	trig_data *proto, *live_trig, *next_trig, *trig = GET_OLC_TRIGGER(desc);
	trig_vnum vnum = GET_OLC_VNUM(desc);
	struct cmdlist_element *cmd, *next_cmd, *cmdlist;
	bool free_text = FALSE;
	UT_hash_handle hh;
	
	// have a place to save it?
	if (!(proto = real_trigger(vnum))) {
		proto = create_trigger_table_entry(vnum);
	}
	
	// build new cmdlist
	cmdlist = compile_command_list(script_text);
	
	// update live triggers
	DL_FOREACH_SAFE2(trigger_list, live_trig, next_trig, next_in_world) {
		if (GET_TRIG_VNUM(live_trig) != vnum) {
			continue;	// wrong trigger
		}
		
		// find any 'waiting' copies and kill them
		if (GET_TRIG_WAIT(live_trig)) {
			dg_event_cancel(GET_TRIG_WAIT(live_trig), cancel_wait_event);
			GET_TRIG_WAIT(live_trig) = NULL;
			GET_TRIG_DEPTH(live_trig) = 0;
			free_varlist(GET_TRIG_VARS(live_trig));
			GET_TRIG_VARS(live_trig) = NULL;
			live_trig->curr_state = NULL;
		}
		
		// check pointers
		if (live_trig->name == proto->name) {
			live_trig->name = trig->name;
		}
		if (live_trig->arglist == proto->arglist) {
			live_trig->arglist = trig->arglist;
		}
		else if (!strcmp(live_trig->arglist, proto->arglist)) {
			free(live_trig->arglist);
			live_trig->arglist = trig->arglist ? str_dup(trig->arglist) : NULL;
		}
		if (live_trig->cmdlist == proto->cmdlist) {
			live_trig->cmdlist = cmdlist;
		}
		if (GET_TRIG_LINKS(live_trig) == GET_TRIG_LINKS(proto)) {
			GET_TRIG_LINKS(live_trig) = GET_TRIG_LINKS(trig);
		}
		
		// check vars
		live_trig->attach_type = trig->attach_type;
		live_trig->trigger_type = trig->trigger_type;
		live_trig->narg = trig->narg;
		
		// update script types on attached-to
		if (live_trig->attached_to) {
			update_script_types(live_trig->attached_to);
		}
	}
	
	// free existing commands
	for (cmd = proto->cmdlist; cmd; cmd = next_cmd) { 
		next_cmd = cmd->next;
		if (cmd->cmd)
			free(cmd->cmd);
		free(cmd);
	}

	// free old data on the proto
	if (proto->arglist) {
		free(proto->arglist);
	}
	if (proto->name) {
		free(proto->name);
	}
	if (proto->var_list) {
		free_varlist(proto->var_list);
	}
	free_trigger_links(&GET_TRIG_LINKS(proto));
	
	if (!*script_text) {
		// do not free old script text
		script_text = str_dup("%echo% This trigger commandlist is not complete!\r\n");
		free_text = TRUE;
	}
	
	// Recompile the command list from the new script
	trig->cmdlist = cmdlist;
	
	if (free_text) {
		free(script_text);
		script_text = NULL;
	}

	// make the prorotype look like what we have
	hh = proto->hh;	// preserve hash handle
	*proto = *trig;
	proto->hh = hh;
	proto->vnum = vnum;	// ensure correct vnum
	
	save_library_file_for_vnum(DB_BOOT_TRG, vnum);
}


/**
* Creates a copy of a trigger, or clears a new one, for editing.
* 
* @param trig_data *input The trigger to copy, or NULL to make a new one.
* @param char **cmdlist_storage A place to store the command list e.g. &GET_OLC_STORAGE(ch->desc)
* @return trig_data* The copied trigger.
*/
trig_data *setup_olc_trigger(trig_data *input, char **cmdlist_storage) {
	struct cmdlist_element *c;
	trig_data *new;
	
	CREATE(*cmdlist_storage, char, MAX_CMD_LENGTH);
	CREATE(new, trig_data, 1);
	trig_data_init(new);
	
	if (input) {
		*new = *input;
		
		// don't copy next/list pointers
		new->next = NULL;
		new->next_in_world = new->prev_in_world = NULL;
		new->next_in_random_triggers = new->prev_in_random_triggers = NULL;
		new->in_world_list = FALSE;
		new->in_random_list = FALSE;
		
		new->name = str_dup(NULLSAFE(input->name));
		new->arglist = input->arglist ? str_dup(input->arglist) : NULL;
		GET_TRIG_LINKS(new) = copy_trigger_links(GET_TRIG_LINKS(input));
		
		// don't copy these things
		new->cmdlist = NULL;
		new->curr_state = NULL;
		new->wait_event = NULL;
		new->var_list = NULL;

		// convert cmdlist to a char string
		c = input->cmdlist;
		strcpy(*cmdlist_storage, "");

		while (c) {
			strcat(*cmdlist_storage, c->cmd);
			strcat(*cmdlist_storage, "\r\n");
			c = c->next;
		}
		// now the cmdlist is something to pass to the text editor
		// it will be converted back to a real cmdlist_element list later
	}
	else {
		new->vnum = NOTHING;

		// Set up some defaults
		new->name = strdup(default_trig_name);
		new->attach_type = MOB_TRIGGER;
		new->trigger_type = NOBITS;

		// cmdlist will be a large char string until the trigger is saved
		strncpy(*cmdlist_storage, "", MAX_CMD_LENGTH-1);
		new->narg = 100;
	}
	
	// done
	return new;	
}


/**
* Counts the words of text in a trigger's strings.
*
* @param trigger_data *trig The trigger whose strings to count.
* @return int The number of words in the trigger's strings.
*/
int wordcount_trigger(trig_data *trig) {
	int count = 0, iter;
	struct cmdlist_element *cmd;
	
	const char *accept_list[] = { "%echo", "%send%", "%mod%", "%regionecho", "%subecho%", "%vehicleecho", "%buildingecho", "say ", "emote ", "shout ", "\n" };
	
	// not player-facing
	// count += wordcount_string(GET_TRIG_NAME(trig));
	// count += wordcount_string(GET_TRIG_ARG(trig));
	
	LL_FOREACH(trig->cmdlist, cmd) {
		for (iter = 0; *accept_list[iter] != '\n'; ++iter) {
			if (cmd->cmd && strstr(cmd->cmd, accept_list[iter])) {
				count += wordcount_string(cmd->cmd) - 1;
				break;
			}
		}
	}
	
	return count;
}


 //////////////////////////////////////////////////////////////////////////////
//// DISPLAYS ////////////////////////////////////////////////////////////////

/**
* This is the main display for trigger OLC. It displays the user's
* currently-edited trigger.
*
* @param char_data *ch The person who is editing a trigger and will see its display.
*/
void olc_show_trigger(char_data *ch) {
	trig_data *trig = GET_OLC_TRIGGER(ch->desc);
	bitvector_t trig_arg_types = compile_argument_types_for_trigger(trig);
	char trgtypes[256], buf[256];
	
	if (!trig) {
		return;
	}
	
	build_page_display(ch, "[%s%d\t0] %s%s\t0", OLC_LABEL_CHANGED, GET_OLC_VNUM(ch->desc), OLC_LABEL_UNCHANGED, !real_trigger(GET_OLC_VNUM(ch->desc)) ? "new trigger" : GET_TRIG_NAME(real_trigger(GET_OLC_VNUM(ch->desc))));
	build_page_display(ch, "<%sname\t0> %s", OLC_LABEL_STR(GET_TRIG_NAME(trig), default_trig_name), NULLSAFE(GET_TRIG_NAME(trig)));
	build_page_display(ch, "<%sattaches\t0> %s", OLC_LABEL_VAL(trig->attach_type, 0), trig_attach_types[trig->attach_type]);
	
	sprintbit(GET_TRIG_TYPE(trig), trig_attach_type_list[trig->attach_type], trgtypes, TRUE);
	build_page_display(ch, "<%stypes\t0> %s", OLC_LABEL_VAL(GET_TRIG_TYPE(trig), NOBITS), trgtypes);
	
	if (IS_SET(trig_arg_types, TRIG_ARG_PERCENT)) {
		build_page_display(ch, "<%spercent\t0> %d%%", OLC_LABEL_VAL(trig->narg, 0), trig->narg);
	}
	if (IS_SET(trig_arg_types, TRIG_ARG_PHRASE_OR_WORDLIST)) {
		build_page_display(ch, "<%sargtype\t0> %s", OLC_LABEL_VAL(trig->narg, 0), trig_arg_phrase_type[trig->narg]);
	}
	if (IS_SET(trig_arg_types, TRIG_ARG_OBJ_WHERE)) {
		sprintbit(trig->narg, trig_arg_obj_where, buf1, TRUE);
		build_page_display(ch, "<%slocation\t0> %s", OLC_LABEL_VAL(trig->narg, 0), trig->narg ? buf1 : "none");
	}
	if (IS_SET(trig_arg_types, TRIG_ARG_COMMAND | TRIG_ARG_PHRASE_OR_WORDLIST)) {
		build_page_display(ch, "<%sstring\t0> %s", OLC_LABEL_STR(trig->arglist, ""), NULLSAFE(trig->arglist));
	}
	if (IS_SET(trig_arg_types, TRIG_ARG_COST)) {
		build_page_display(ch, "<%scosts\t0> %d misc coins", OLC_LABEL_VAL(trig->narg, 0), trig->narg);
	}
	
	safe_snprintf(buf, sizeof(buf), "<%slinks\t0>", OLC_LABEL_PTR(GET_TRIG_LINKS(trig)));
	build_trigger_link_page_display(ch, trig, buf);
	
	build_page_display(ch, "<%scommands\t0>\r\n%s", OLC_LABEL_STR(GET_OLC_STORAGE(ch->desc), ""), show_color_codes(NULLSAFE(GET_OLC_STORAGE(ch->desc))));
	
	send_page_display(ch);
}


/**
* Builds a list of trigger links into the player's page display.
*
* @param char_data *ch The player viewing it.
* @param trig_data *trig Which trigger's links to show.
* @param const char *header Optional: A header shown first (may be empty or NULL).
*/
void build_trigger_link_page_display(char_data *ch, trig_data *trig, const char *header) {
	bitvector_t last_type = NOBITS;
	char buf[MAX_STRING_LENGTH];
	int count = 0, width;
	struct page_display *line = NULL;
	struct trig_link *link;
	
	width = (ch && ch->desc && ch->desc->pProtocol->ScreenWidth > 15) ? (ch->desc->pProtocol->ScreenWidth - 1) : 79;
	
	LL_SORT(GET_TRIG_LINKS(trig), sort_trigger_links);
	if (header && *header) {
		build_page_display_str(ch, header);
	}
	
	LL_FOREACH(GET_TRIG_LINKS(trig), link) {
		// header
		if (last_type != link->type || !line) {
			last_type = link->type;
			prettier_sprintbit(link->type, olc_type_bits, buf);
			line = build_page_display(ch, " %s: ", CAP(buf));
			count = 0;
		}
		
		// build display, check wrap, append
		safe_snprintf(buf, sizeof(buf), "[%d] %s", link->vnum, get_name_by_olc_type(link->type, link->vnum));
		if (line->length + strlen(buf) + 2 > width) {
			if (count++ > 0) {
				append_page_display_line(line, ",");
			}
			line = build_page_display(ch, "  %s", buf);
		}
		else {
			append_page_display_line(line, "%s%s", (count++ > 0 ? ", " : ""), buf); 
		}
	}
	
	// only show "none" if we show a header
	if (count == 0 && header && *header) {
		build_page_display_str(ch, "  none");
	}
}


 //////////////////////////////////////////////////////////////////////////////
//// EDIT MODULES ////////////////////////////////////////////////////////////

OLC_MODULE(tedit_argtype) {
	trig_data *trig = GET_OLC_TRIGGER(ch->desc);
	bitvector_t trig_arg_types = compile_argument_types_for_trigger(trig);
	
	if (!IS_SET(trig_arg_types, TRIG_ARG_PHRASE_OR_WORDLIST)) {
		msg_to_char(ch, "You can't set that property on this trigger.\r\n");
	}
	else {
		GET_TRIG_NARG(trig) = olc_process_type(ch, argument, "argument type", "argtype", trig_arg_phrase_type, GET_TRIG_NARG(trig));
	}
}


OLC_MODULE(tedit_attaches) {
	trig_data *trig = GET_OLC_TRIGGER(ch->desc);
	int old = trig->attach_type;
	
	trig->attach_type = olc_process_type(ch, argument, "attach type", "attaches", trig_attach_types, trig->attach_type);
	
	if (old != trig->attach_type) {
		GET_TRIG_TYPE(trig) = NOBITS;
	}
}


OLC_MODULE(tedit_commands) {
	if (ch->desc->str) {
		msg_to_char(ch, "You are already editing a string.\r\n");
	}
	else {
		start_string_editor(ch->desc, "trigger commands", &GET_OLC_STORAGE(ch->desc), MAX_CMD_LENGTH, FALSE);
	}
}


OLC_MODULE(tedit_costs) {
	trig_data *trig = GET_OLC_TRIGGER(ch->desc);
	bitvector_t trig_arg_types = compile_argument_types_for_trigger(trig);
	
	if (!IS_SET(trig_arg_types, TRIG_ARG_COST)) {
		msg_to_char(ch, "You can't set that property on this trigger.\r\n");
	}
	else {
		GET_TRIG_NARG(trig) = olc_process_number(ch, argument, "cost", "costs", 0, MAX_INT, GET_TRIG_NARG(trig));
	}
}


OLC_MODULE(tedit_links) {
	trig_data *trig = GET_OLC_TRIGGER(ch->desc);
	any_vnum vnum;
	bool any;
	char cmd_arg[MAX_INPUT_LENGTH], type_arg[MAX_INPUT_LENGTH], vnum_arg[MAX_INPUT_LENGTH], buf[MAX_STRING_LENGTH];
	int pos;
	struct trig_link *link, *next_link;
	
	const char *usage = "Usage: links add <type> <vnum>\r\n"
						"       links remove <type> <vnum | all>\r\n"
						"       links remove all\r\n";
	
	argument = any_one_arg(argument, cmd_arg);
	argument = any_one_arg(argument, type_arg);
	argument = any_one_arg(argument, vnum_arg);
	
	if (!*cmd_arg || !*type_arg) {
		msg_to_char(ch, "%s", usage);
	}
	else if (is_abbrev(cmd_arg, "add")) {
		if (!*vnum_arg || !isdigit(*vnum_arg) || (vnum = atoi(vnum_arg)) < 0) {
			msg_to_char(ch, "%s", usage);
		}
		else if ((pos = search_block(type_arg, olc_type_bits, FALSE)) == NOTHING) {
			msg_to_char(ch, "Unknown type '%s'.\r\n", type_arg);
		}
		else {
			// ok to add -- first ensure not already in list
			any = FALSE;
			LL_FOREACH(GET_TRIG_LINKS(trig), link) {
				if (IS_SET(link->type, BIT(pos)) && link->vnum == vnum) {
					any = TRUE;
					break;
				}
			}
			
			prettier_sprintbit(BIT(pos), olc_type_bits, buf);
			if (any) {
				msg_to_char(ch, "Link already exists to %s [%d] %s.\r\n", buf, vnum, get_name_by_olc_type(BIT(pos), vnum));
			}
			else {
				CREATE(link, struct trig_link, 1);
				link->type = BIT(pos);
				link->vnum = vnum;
				LL_PREPEND(GET_TRIG_LINKS(trig), link);
				LL_SORT(GET_TRIG_LINKS(trig), sort_trigger_links);
				msg_to_char(ch, "Link added for %s [%d] %s.\r\n", buf, vnum, get_name_by_olc_type(BIT(pos), vnum));
			}
		}
	}
	else if (is_abbrev(cmd_arg, "remove")) {
		if (!str_cmp(type_arg, "all")) {
			free_trigger_links(&GET_TRIG_LINKS(trig));
			msg_to_char(ch, "You remove all the links.\r\n");
		}
		else if (!*vnum_arg || (!isdigit(*vnum_arg) && str_cmp(vnum_arg, "all")) || (vnum = atoi(vnum_arg)) < 0) {
			msg_to_char(ch, "You must specify a vnum to remove (or 'all').\r\n");
		}
		else if ((pos = search_block(type_arg, olc_type_bits, FALSE)) == NOTHING) {
			msg_to_char(ch, "Unknown type '%s'.\r\n", type_arg);
		}
		else {
			// remove by type
			any = FALSE;
			LL_FOREACH_SAFE(GET_TRIG_LINKS(trig), link, next_link) {
				if (IS_SET(link->type, BIT(pos)) && (link->vnum == vnum || !str_cmp(vnum_arg, "all"))) {
					LL_DELETE(GET_TRIG_LINKS(trig), link);
					free(link);
					any = TRUE;
				}
			}
			
			prettier_sprintbit(BIT(pos), olc_type_bits, buf);
			if (!any) {
				msg_to_char(ch, "No matching links found to remove.\r\n");
			}
			else if (!str_cmp(vnum_arg, "all")) {
				msg_to_char(ch, "Removed all %s links.\r\n", buf);
			}
			else {
				msg_to_char(ch, "Removed link to %s [%d] %s.\r\n", buf, vnum, get_name_by_olc_type(BIT(pos), vnum));
			}
		}
	}
	else {
		msg_to_char(ch, "%s", usage);
	}
}


OLC_MODULE(tedit_location) {
	trig_data *trig = GET_OLC_TRIGGER(ch->desc);
	bitvector_t trig_arg_types = compile_argument_types_for_trigger(trig);
	
	if (!IS_SET(trig_arg_types, TRIG_ARG_OBJ_WHERE)) {
		msg_to_char(ch, "You can't set that property on this trigger.\r\n");
	}
	else {
		GET_TRIG_NARG(trig) = olc_process_flag(ch, argument, "location", "location", trig_arg_obj_where, GET_TRIG_NARG(trig));
	}
}


OLC_MODULE(tedit_name) {
	trig_data *trig = GET_OLC_TRIGGER(ch->desc);
	olc_process_string(ch, argument, "name", &GET_TRIG_NAME(trig));
}


OLC_MODULE(tedit_numarg) {
	trig_data *trig = GET_OLC_TRIGGER(ch->desc);
	GET_TRIG_NARG(trig) = olc_process_number(ch, argument, "numeric argument", "numarg", -MAX_INT, MAX_INT, GET_TRIG_NARG(trig));
}


OLC_MODULE(tedit_percent) {
	trig_data *trig = GET_OLC_TRIGGER(ch->desc);
	bitvector_t trig_arg_types = compile_argument_types_for_trigger(trig);
	
	if (!IS_SET(trig_arg_types, TRIG_ARG_PERCENT)) {
		msg_to_char(ch, "You can't set that property on this trigger.\r\n");
	}
	else {
		GET_TRIG_NARG(trig) = olc_process_number(ch, argument, "percent", "percent", 0, 100, GET_TRIG_NARG(trig));
	}
}


OLC_MODULE(tedit_string) {
	trig_data *trig = GET_OLC_TRIGGER(ch->desc);
	bitvector_t trig_arg_types = compile_argument_types_for_trigger(trig);
	
	if (!IS_SET(trig_arg_types, TRIG_ARG_COMMAND | TRIG_ARG_PHRASE_OR_WORDLIST)) {
		msg_to_char(ch, "You can't set that property on this trigger.\r\n");
	}
	else {
		olc_process_string(ch, argument, "string", &GET_TRIG_ARG(trig));
	}
}


OLC_MODULE(tedit_types) {
	trig_data *trig = GET_OLC_TRIGGER(ch->desc);	
	bitvector_t old = GET_TRIG_TYPE(trig), diff;
	
	bitvector_t ignore_changes = MTRIG_GLOBAL | MTRIG_PLAYER_IN_ROOM | MTRIG_ALLOW_MULTIPLE;	// all types ignore global changes
	
	// mobs also ignore changes to the charmed flag
	if (trig->attach_type == MOB_TRIGGER) {
		ignore_changes |= MTRIG_CHARMED;
	}
	
	GET_TRIG_TYPE(trig) = olc_process_flag(ch, argument, "type", "types", trig_attach_type_list[trig->attach_type], GET_TRIG_TYPE(trig));
	
	diff = (old & ~GET_TRIG_TYPE(trig)) | (~old & GET_TRIG_TYPE(trig));	// find changed bits
	diff &= ~ignore_changes;	// filter out ones we don't care about
	
	// if we changed any relevant type, remove args
	if (diff != NOBITS) {
		trig->narg = 0;
		if (trig->arglist) {
			free(trig->arglist);
			trig->arglist = NULL;
			msg_to_char(ch, "Type changed. Arguments cleared.\r\n");
		}
	}
}
