#ifndef UTILS_STRING_HPP_
#define UTILS_STRING_HPP_

#include <cstring>
#include <memory>
#include <string>
#include <vector>

namespace utils {
using shared_string_ptr = std::shared_ptr<char>;

class Padding {
 public:
	Padding(const std::string &string, std::size_t minimal_length, const char padding = ' ') :
		m_length(string.size() > minimal_length ? 0 : minimal_length - string.size()),
		m_padding(padding) {
	}

	[[nodiscard]] auto length() const { return m_length; }
	std::ostream &output(std::ostream &os) const;

 protected:
	std::ostream &pad(std::ostream &os, const std::size_t length) const;

 private:
	std::size_t m_length;
	char m_padding;
};

inline std::ostream &operator<<(std::ostream &os, const Padding &padding) { return padding.output(os); }

class SpacedPadding : public Padding {
 public:
	SpacedPadding(const std::string &string, std::size_t minimal_length, const char padding = ' ') :
		Padding(string, minimal_length, padding) {
	}

	std::ostream &output(std::ostream &os) const;
};

inline std::ostream &operator<<(std::ostream &os, const SpacedPadding &padding) { return padding.output(os); }

class AbstractStringWriter {
 public:
	using shared_ptr = std::shared_ptr<AbstractStringWriter>;

	virtual ~AbstractStringWriter() = default;
	[[nodiscard]] virtual const char *get_string() const = 0;
	virtual void set_string(const char *data) = 0;
	virtual void append_string(const char *data) = 0;
	[[nodiscard]] virtual size_t length() const = 0;
	virtual void clear() = 0;
};

class DelegatedStringWriter : public AbstractStringWriter {
 public:
	explicit DelegatedStringWriter(char *&managed) : m_delegated_string_(managed) {}
	[[nodiscard]] const char *get_string() const final { return m_delegated_string_; }
	void set_string(const char *string) final;
	void append_string(const char *string) final;
	[[nodiscard]] size_t length() const final { return m_delegated_string_ ? strlen(m_delegated_string_) : 0; }
	void clear() override;

 private:
	char *&m_delegated_string_;
};

class AbstractStdStringWriter : public AbstractStringWriter {
 public:
	[[nodiscard]] const char *get_string() const override { return string().c_str(); }
	void set_string(const char *string) override { this->string() = string; }
	void append_string(const char *string) override { this->string() += string; }
	[[nodiscard]] size_t length() const override { return string().length(); }
	void clear() override { string().clear(); }

 private:
	virtual std::string &string() = 0;
	[[nodiscard]] virtual const std::string &string() const = 0;
};

class StdStringWriter : public AbstractStdStringWriter {
 private:
	virtual std::string &string() override { return m_string_; }
	virtual const std::string &string() const override { return m_string_; }

	std::string m_string_;
};

class DelegatedStdStringWriter : public AbstractStringWriter {
 public:
	DelegatedStdStringWriter(std::string &string) : m_string_(string) {}
	virtual const char *get_string() const override { return m_string_.c_str(); }
	virtual void set_string(const char *string) override { m_string_ = string; }
	virtual void append_string(const char *string) override { m_string_ += string; }
	virtual size_t length() const override { return m_string_.length(); }
	virtual void clear() override { m_string_.clear(); }

 private:
	std::string &m_string_;
};

std::string RemoveColors(char *string);
std::string RemoveColors(std::string string);
shared_string_ptr GetStringWithoutColors(const char *string);
std::string GetStringWithoutColors(const std::string &string);

bool IsAbbr(const char *arg1, const char *arg2);
inline int IsAbbrev(const std::string &arg1, const char *arg2) { return IsAbbr(arg1.c_str(), arg2); }


//    
void ConvertToLow(std::string &text);
void ConvertToLow(char *text);

/**
 *           tokens.
 * @param tokens -  .
 * @param s -  .
 * @param delimiter - -,   - .
 */
std::vector<std::string> Split(const std::string s, char delimiter = ' ');

/**
 *           tokens.
 *  	boost::split(option_list, options, boost::is_any_of(", "), boost::token_compress_on);
 * @param tokens -  .
 * @param s -  .
 * @param any - -,  ",- "
 */
std::vector<std::string> SplitAny(const std::string s, std::string any);

//  one_argument  string
// s -  
//   ,  remains ,      
std::string ExtractFirstArgument(const std::string &s, std::string &remains);

//    
std::string FirstWordOnString(std::string s, std::string mask);

/**
 *   .
 */
void TrimLeft(std::string &s);
void TrimLeft(char *s);

/**
 *    .
 */
void TrimRight(std::string &s);
void TrimRight(char* string);
/**
 *     .
 */
void Trim(std::string &s);
void Trim(char *s);

/**
 *   ,  .
 */
std::string TrimLeftCopy(std::string s);

/**
 *    ,  .
 */
std::string TrimRightCopy(std::string s);

/**
 *     ,  .
 */
std::string TrimCopy(std::string s);

/**
 *        .
 *      ,    Trim.
 * @param s -  .
 * @param whitespaces -  ,   .
 */
void TrimLeftIf(std::string &s, const std::string &whitespaces);

/**
 *        .
 *      ,    Trim.
 * @param s -  .
 * @param whitespaces -  ,   .
 */
void TrimRightIf(std::string &s, const std::string &whitespaces);

/**
 *          .
 *      ,    Trim.
 * @param s -  .
 * @param whitespaces -  ,   .
 */
void TrimIf(std::string &s, const std::string &whitespaces);

/**
 *  Koi  WIN.
 */
void ConvertKtoW(std::string &text);

/**
 *  WIN  Koi.
 */
void ConvertWtoK(std::string &text);

/**
 *  Koi  WIN.  .
 */
std::string SubstKtoW(std::string s);

/**
 *  WIN  Koi.  .
 */
std::string SubstWtoK(std::string s);

/**
 *  KOI8-R ,  (-).
 */
void SortKoiString(std::vector<std::string> &str);

/**
 *  KOI8-R ,  (-).
 */
void SortKoiStringReverse(std::vector<std::string> &str);

/**
 *  '.'  '_'  .
 */
std::string FixDot(std::string s);

/**
 *     .
 */
std::string SubstStrToLow(std::string s);

/**
 *     .
 */
std::string SubstStrToUpper(std::string s);


/**
 *        .
 * @param s -  .
 * @param toSearch -  .
 * @param replacer - -.
 */
void ReplaceFirst(std::string &s, const std::string &toSearch, const std::string &replacer);

/**
 *       .
 * @param s -  .
 * @param toSearch -  .
 * @param replacer - -.
 */
void ReplaceAll(std::string &s, const std::string &toSearch, const std::string &replacer);

//   ,    
void ReplaceTrigerNumber(std::string &s, const std::string &toSearch, const std::string &replacer);
/**
 *     .
 * @param s -  .
 * @param toSearch -  .
 */
void EraseAll(std::string &s, const std::string &toSearch);

/**
 *      .
 * @param s -  .
 * @param any -  .
 */
std::string EraseAllAny(const std::string s, const std::string any);

//    s    ch
std::string CompressSymbol(std::string s, const char ch);

//      
char *colorLOW(char *txt);
std::string &colorLOW(std::string &txt);
std::string &colorLOW(std::string &&txt);
//     
char *colorCAP(char *txt);
std::string &colorCAP(std::string &txt);
std::string &colorCAP(std::string &&txt);
//   
char *CAP(char *txt);
std::string CAP(const std::string txt);

} // namespace utils

#endif // UTILS_STRING_HPP_

// vim: ts=4 sw=4 tw=0 noet syntax=cpp :
