/****************************[14~*********************************************
*   File: liquid.cpp                                   Part of Bylins    *
*                                                          *
*                                                                        *
*  $Author$                                                      *
*  $Date$                                          *
*  $Revision$                                                     *
************************************************************************ */

#include "liquid.h"

#include "engine/entities/obj_data.h"
#include "engine/entities/char_data.h"
#include "engine/core/utils_char_obj.inl"
#include "gameplay/magic/magic.h"
#include "engine/ui/color.h"
#include "engine/db/global_objects.h"
#include "poison.h"

#include <cmath>

const int kDrunked = 10;
const int kMortallyDrunked = 18;
const int kMaxCondition = 48;
const int kNormCondition = 22;

const char *drinks[] = {"",
						"",
						"",
						"",
						"",
						"",
						"",
						"",
						"",
						"",
						"",
						"",
						"",
						"",
						" ",
						" ",
						" ",
						"  ",
						"  ",
						"  ",
						"  ", //20
						"  ",
						"  ",
						"  ",
						"  ",
						" ",
						" ",
						" ",
						" ",
						" ",
						"\n"
};

// one-word alias for each drink
const char *drinknames[] = {"",
							"",
							"",
							"",
							"",
							"",
							"",
							"",
							"",
							"",
							"",
							"",
							"",
							"",
							" ",
							" ",
							" ",
							"  ",
							"  ",
							"  ",
							"  ", //20
							"  ",
							"  ",
							"  ",
							"  ",
							" ",
							" ",
							" ",
							" ",
							" ",
							"\n"
};

// color of the various drinks
const char *color_liquid[] = {"",
							  "",
							  "",
							  "",
							  "",
							  "",
							  "",
							  "",
							  "",
							  " ",
							  "",
							  "",
							  "-",
							  "",
							  "",
							  " ",
							  "",
							  "",
							  "",
							  "",
							  "", //20
							  " ",
							  "",
							  "",
							  "",
							  "",
							  "",
							  "",
							  "",
							  "\n"
};

// effect of drinks on DRUNK, FULL, THIRST -- see values.doc
const int drink_aff[][3] = {
	{0, 1, -10},            // 
	{2, -2, -3},            // 
	{5, -2, -2},            // 
	{3, -2, -3},            // 
	{1, -2, -5},            // 
	{8, 0,
	 4},                //  (   ,         ,   )
	{0, -1, -8},            // 
	{10, 0,
	 3},                //  (  !   ,    ,   )
	{3, -3, -3},            // 
	{0, -2, -8},            //  (   )
	{0, -3, -6},            // 
	{0, -1, -6},            // 
	{0, -1, -6},            // 
	{0, -2, 1},            // 
	{0, -1, 2},            //  
	{0, 0, -13},            //  
	{0, -1, 1},            //  
	{0, -1, 1},            //   
	{0, -1, 1},            //   
	{0, -1, 1},            //   
	{0, -1, 1},            //   
	{0, -1, 1},            //   
	{0, -1, 1},            //   
	{0, -1, 1},            //   
	{0, -1, 1},            //   
	{0, 0, 0},            //  
	{0, 0, 0},            //  
	{0, 0, 0},            //  
	{0, 0, 0}            //  
};

/**
* ,   ,     .
*  ,     , 
*       .
*/
bool is_potion(const ObjData *obj) {
	switch (GET_OBJ_VAL(obj, 2)) {
		case LIQ_POTION:
		case LIQ_POTION_RED:
		case LIQ_POTION_BLUE:
		case LIQ_POTION_WHITE:
		case LIQ_POTION_GOLD:
		case LIQ_POTION_BLACK:
		case LIQ_POTION_GREY:
		case LIQ_POTION_FUCHSIA:
		case LIQ_POTION_PINK: return true;
			break;
	}
	return false;
}

namespace drinkcon {

ObjVal::EValueKey init_spell_num(int num) {
	return num == 1
		   ? ObjVal::EValueKey::POTION_SPELL1_NUM
		   : (num == 2
			  ? ObjVal::EValueKey::POTION_SPELL2_NUM
			  : ObjVal::EValueKey::POTION_SPELL3_NUM);
}

ObjVal::EValueKey init_spell_lvl(int num) {
	return num == 1
		   ? ObjVal::EValueKey::POTION_SPELL1_LVL
		   : (num == 2
			  ? ObjVal::EValueKey::POTION_SPELL2_LVL
			  : ObjVal::EValueKey::POTION_SPELL3_LVL);
}

void reset_potion_values(CObjectPrototype *obj) {
	obj->SetPotionValueKey(ObjVal::EValueKey::POTION_SPELL1_NUM, -1);
	obj->SetPotionValueKey(ObjVal::EValueKey::POTION_SPELL1_LVL, -1);
	obj->SetPotionValueKey(ObjVal::EValueKey::POTION_SPELL2_NUM, -1);
	obj->SetPotionValueKey(ObjVal::EValueKey::POTION_SPELL2_LVL, -1);
	obj->SetPotionValueKey(ObjVal::EValueKey::POTION_SPELL3_NUM, -1);
	obj->SetPotionValueKey(ObjVal::EValueKey::POTION_SPELL3_LVL, -1);
	obj->SetPotionValueKey(ObjVal::EValueKey::POTION_PROTO_VNUM, -1);
}

///    (GET_OBJ_VAL(from_obj, 0))     
bool copy_value(const CObjectPrototype *from_obj, CObjectPrototype *to_obj, int num) {
	if (GET_OBJ_VAL(from_obj, num) > 0) {
		to_obj->SetPotionValueKey(init_spell_num(num), GET_OBJ_VAL(from_obj, num));
		to_obj->SetPotionValueKey(init_spell_lvl(num), GET_OBJ_VAL(from_obj, 0));
		return true;
	}
	return false;
}

///  values  (to_obj)   (from_obj)
void copy_potion_values(const CObjectPrototype *from_obj, CObjectPrototype *to_obj) {
	reset_potion_values(to_obj);
	bool copied = false;

	for (int i = 1; i <= 3; ++i) {
		if (copy_value(from_obj, to_obj, i)) {
			copied = true;
		}
	}

	if (copied) {
		to_obj->SetPotionValueKey(ObjVal::EValueKey::POTION_PROTO_VNUM, GET_OBJ_VNUM(from_obj));
	}
}

} // namespace drinkcon

using namespace drinkcon;

int cast_potion_spell(CharData *ch, ObjData *obj, int num) {
	const auto spell_id = static_cast<ESpell>(obj->GetPotionValueKey(init_spell_num(num)));
	const int level = -obj->GetPotionValueKey(init_spell_lvl(num));

	if (spell_id > ESpell::kUndefined) {
		return CallMagic(ch, ch, nullptr, world[ch->in_room], spell_id, level);
	}
	return 1;
}

int TryCastSpellsFromLiquid(CharData *ch, ObjData *jar) {
	if (is_potion(jar) && jar->GetPotionValueKey(ObjVal::EValueKey::POTION_PROTO_VNUM) >= 0) {
		act("$n $g   $o1.", true, ch, jar, 0, kToRoom);
		SendMsgToChar(ch, "    %s.\r\n", OBJN(jar, ch, ECase::kGen));

		//  ,   
		for (int i = 1; i <= 3; ++i)
			if (cast_potion_spell(ch, jar, i) <= 0)
				break;

		SetWaitState(ch, kBattleRound);
		jar->dec_weight();
		//  
		jar->dec_val(1);

		if (GET_OBJ_VAL(jar, 1) <= 0
			&& jar->get_type() != EObjType::kFountain) {
			name_from_drinkcon(jar);
			jar->set_spec_param(0);
			reset_potion_values(jar);
		}
		TryDrinkPoison(ch, jar, 0);
		return 1;
	}
	return 0;
}

void drinkcon::generate_drinkcon_name(ObjData *to_obj, ESpell spell_id) {
	switch (spell_id) {
		//  () //
		case ESpell::kResfresh:
		case ESpell::kGroupRefresh: to_obj->set_val(2, LIQ_POTION_RED);
			name_to_drinkcon(to_obj, LIQ_POTION_RED);
			break;
			//  () //
		case ESpell::kFullFeed:
		case ESpell::kCommonMeal: to_obj->set_val(2, LIQ_POTION_BLUE);
			name_to_drinkcon(to_obj, LIQ_POTION_BLUE);
			break;
			//  () //
		case ESpell::kDetectInvis:
		case ESpell::kAllSeeingEye:
		case ESpell::kDetectMagic:
		case ESpell::kMagicalGaze:
		case ESpell::kDetectPoison:
		case ESpell::kSnakeEyes:
		case ESpell::kDetectAlign:
		case ESpell::kGroupSincerity:
		case ESpell::kSenseLife:
		case ESpell::kEyeOfGods:
		case ESpell::kInfravision:
		case ESpell::kSightOfDarkness: to_obj->set_val(2, LIQ_POTION_WHITE);
			name_to_drinkcon(to_obj, LIQ_POTION_WHITE);
			break;
			//  () //
		case ESpell::kArmor:
		case ESpell::kGroupArmor:
		case ESpell::kCloudly: to_obj->set_val(2, LIQ_POTION_GOLD);
			name_to_drinkcon(to_obj, LIQ_POTION_GOLD);
			break;
			//   () //
		case ESpell::kCureCritic:
		case ESpell::kCureLight:
		case ESpell::kHeal:
		case ESpell::kGroupHeal:
		case ESpell::kCureSerious: to_obj->set_val(2, LIQ_POTION_BLACK);
			name_to_drinkcon(to_obj, LIQ_POTION_BLACK);
			break;
			//    () //
		case ESpell::kCureBlind:
		case ESpell::kRemoveCurse:
		case ESpell::kRemoveHold:
		case ESpell::kRemoveSilence:
		case ESpell::kCureFever:
		case ESpell::kRemoveDeafness:
		case ESpell::kRemovePoison: to_obj->set_val(2, LIQ_POTION_GREY);
			name_to_drinkcon(to_obj, LIQ_POTION_GREY);
			break;
			//   () //
		case ESpell::kInvisible:
		case ESpell::kGroupInvisible:
		case ESpell::kStrength:
		case ESpell::kGroupStrength:
		case ESpell::kFly:
		case ESpell::kGroupFly:
		case ESpell::kBless:
		case ESpell::kGroupBless:
		case ESpell::kHaste:
		case ESpell::kGroupHaste:
		case ESpell::kStoneSkin:
		case ESpell::kStoneWall:
		case ESpell::kBlink:
		case ESpell::kExtraHits:
		case ESpell::kWaterbreath: to_obj->set_val(2, LIQ_POTION_FUCHSIA);
			name_to_drinkcon(to_obj, LIQ_POTION_FUCHSIA);
			break;
		case ESpell::kPrismaticAura:
		case ESpell::kGroupPrismaticAura:
		case ESpell::kAirAura:
		case ESpell::kEarthAura:
		case ESpell::kFireAura:
		case ESpell::kIceAura: to_obj->set_val(2, LIQ_POTION_PINK);
			name_to_drinkcon(to_obj, LIQ_POTION_PINK);
			break;
		default: to_obj->set_val(2, LIQ_POTION);
			name_to_drinkcon(to_obj, LIQ_POTION);    //    //
	}
}

int check_potion_spell(ObjData *from_obj, ObjData *to_obj, int num) {
	const auto spell = init_spell_num(num);
	const auto level = init_spell_lvl(num);

	if (GET_OBJ_VAL(from_obj, num) != to_obj->GetPotionValueKey(spell)) {
		//   
		return 0;
	}
	if (GET_OBJ_VAL(from_obj, 0) < to_obj->GetPotionValueKey(level)) {
		//       
		return -1;
	}
	return 1;
}

/// \return 1 -  
///         0 -    
///        -1 -       
int drinkcon::check_equal_potions(ObjData *from_obj, ObjData *to_obj) {
	//      
	if (to_obj->GetPotionValueKey(ObjVal::EValueKey::POTION_PROTO_VNUM) > 0
		&& GET_OBJ_VNUM(from_obj) != to_obj->GetPotionValueKey(ObjVal::EValueKey::POTION_PROTO_VNUM)) {
		return 0;
	}
	//      
	for (int i = 1; i <= 3; ++i) {
		if (GET_OBJ_VAL(from_obj, i) > 0) {
			int result = check_potion_spell(from_obj, to_obj, i);
			if (result <= 0) {
				return result;
			}
		}
	}
	return 1;
}

///  check_equal_drinkcon()
int check_drincon_spell(ObjData *from_obj, ObjData *to_obj, int num) {
	const auto spell = init_spell_num(num);
	const auto level = init_spell_lvl(num);

	if (from_obj->GetPotionValueKey(spell) != to_obj->GetPotionValueKey(spell)) {
		//   
		return 0;
	}
	if (from_obj->GetPotionValueKey(level) < to_obj->GetPotionValueKey(level)) {
		//       
		return -1;
	}
	return 1;
}

///   check_equal_potions   ,   
///   values  /.
/// \return 1 -  
///         0 -    
///        -1 -       
int drinkcon::check_equal_drinkcon(ObjData *from_obj, ObjData *to_obj) {
	//       (    ,  )
	for (int i = 1; i <= 3; ++i) {
		if (GET_OBJ_VAL(from_obj, i) > 0) {
			int result = check_drincon_spell(from_obj, to_obj, i);
			if (result <= 0) {
				return result;
			}
		}
	}
	return 1;
}

///            
void drinkcon::spells_to_drinkcon(ObjData *from_obj, ObjData *to_obj) {
	//  
	for (int i = 1; i <= 3; ++i) {
		const auto spell = init_spell_num(i);
		const auto level = init_spell_lvl(i);
		to_obj->SetPotionValueKey(spell, from_obj->GetPotionValueKey(spell));
		to_obj->SetPotionValueKey(level, from_obj->GetPotionValueKey(level));
	}
	//      
	const int proto_vnum = from_obj->GetPotionValueKey(ObjVal::EValueKey::POTION_PROTO_VNUM) > 0
						   ? from_obj->GetPotionValueKey(ObjVal::EValueKey::POTION_PROTO_VNUM)
						   : GET_OBJ_VNUM(from_obj);
	to_obj->SetPotionValueKey(ObjVal::EValueKey::POTION_PROTO_VNUM, proto_vnum);
}

size_t find_liquid_name(const char *name) {
	std::string tmp = std::string(name);
	size_t pos, result = std::string::npos;
	for (int i = 0; strcmp(drinknames[i], "\n"); i++) {
		pos = tmp.find(drinknames[i]);
		if (pos != std::string::npos) {
			result = pos;
		}
	}
	return result;
}

void name_from_drinkcon(ObjData *obj) {
	char new_name[kMaxStringLength];
	std::string tmp;

	size_t pos = find_liquid_name(obj->get_aliases().c_str());
	if (pos == std::string::npos) return;
	tmp = obj->get_aliases().substr(0, pos - 1);

	sprintf(new_name, "%s", tmp.c_str());
	obj->set_aliases(new_name);

	pos = find_liquid_name(obj->get_short_description().c_str());
	if (pos == std::string::npos) return;
	tmp = obj->get_short_description().substr(0, pos - 3);

	sprintf(new_name, "%s", tmp.c_str());
	obj->set_short_description(new_name);

	for (int c = ECase::kFirstCase; c <= ECase::kLastCase; c++) {
		auto name_case = static_cast<ECase>(c);
		pos = find_liquid_name(obj->get_PName(name_case).c_str());
		if (pos == std::string::npos) return;
		tmp = obj->get_PName(name_case).substr(0, pos - 3);
		sprintf(new_name, "%s", tmp.c_str());
		obj->set_PName(name_case, new_name);
	}
}

void name_to_drinkcon(ObjData *obj, int type) {
	int c;
	char new_name[kMaxInputLength], potion_name[kMaxInputLength];
	if (type >= NUM_LIQ_TYPES) {
		snprintf(potion_name, kMaxInputLength, "%s", " ,  ");
	} else {
		snprintf(potion_name, kMaxInputLength, "%s", drinknames[type]);
	}

	snprintf(new_name, kMaxInputLength, "%s %s", obj->get_aliases().c_str(), potion_name);
	obj->set_aliases(new_name);
	snprintf(new_name, kMaxInputLength, "%s  %s", obj->get_short_description().c_str(), potion_name);
	obj->set_short_description(new_name);

	for (c = ECase::kFirstCase; c <= ECase::kLastCase; c++) {
		auto name_case = static_cast<ECase>(c);
		snprintf(new_name, kMaxInputLength, "%s  %s", obj->get_PName(name_case).c_str(), potion_name);
		obj->set_PName(name_case, new_name);
	}
}

std::string print_spell(const ObjData *obj, int num) {
	const auto spell = init_spell_num(num);
	const auto level = init_spell_lvl(num);

	if (obj->GetPotionValueKey(spell) == -1) {
		return "";
	}

	char buf_[kMaxInputLength];
	snprintf(buf_, sizeof(buf_), " : %s%s (%d .)%s\r\n",
			 kColorCyn,
			 MUD::Spell(static_cast<ESpell>(obj->GetPotionValueKey(spell))).GetCName(),
			 obj->GetPotionValueKey(level),
			 kColorNrm);

	return buf_;
}

namespace drinkcon {

std::string print_spells(const ObjData *obj) {
	std::string out;
	char buf_[kMaxInputLength];

	for (int i = 1; i <= 3; ++i) {
		out += print_spell(obj, i);
	}

	if (!out.empty() && !is_potion(obj)) {
		snprintf(buf_, sizeof(buf_), "%s%s:     \r\n",
				 kColorBoldRed, kColorNrm);
		out += buf_;
	} else if (out.empty() && is_potion(obj)) {
		snprintf(buf_, sizeof(buf_), "%s%s:     \r\n",
				 kColorBoldRed, kColorNrm);
		out += buf_;
	}

	return out;
}

void identify(CharData *ch, const ObjData *obj) {
	std::string out;
	char buf_[kMaxInputLength];
	int volume = GET_OBJ_VAL(obj, 0);
	int amount = GET_OBJ_VAL(obj, 1);

	snprintf(buf_, sizeof(buf_), "  : %s%d %s%s\r\n",
			 kColorCyn,
			 volume, GetDeclensionInNumber(volume, EWhat::kGulp),
			 kColorNrm);
	out += buf_;

	//   
	if (amount > 0) {
		//  - 
		if (obj->GetPotionValueKey(ObjVal::EValueKey::POTION_PROTO_VNUM) >= 0) {
			if (IS_IMMORTAL(ch)) {
				snprintf(buf_, sizeof(buf_), " %d %s %s (VNUM: %d).\r\n",
						 amount,
						 GetDeclensionInNumber(amount, EWhat::kGulp),
						 drinks[GET_OBJ_VAL(obj, 2)],
						 obj->GetPotionValueKey(ObjVal::EValueKey::POTION_PROTO_VNUM));
			} else {
				snprintf(buf_, sizeof(buf_), " %d %s %s.\r\n",
						 amount,
						 GetDeclensionInNumber(amount, EWhat::kGulp),
						 drinks[GET_OBJ_VAL(obj, 2)]);
			}
			out += buf_;
			out += print_spells(obj);
		} else {
			snprintf(buf_, sizeof(buf_), "%s %s  %d%%\r\n",
					 GET_OBJ_SUF_6(obj),
					 drinknames[GET_OBJ_VAL(obj, 2)],
					 amount * 100 / (volume ? volume : 1));
			out += buf_;
			//        
			if (is_potion(obj)) {
				out += print_spells(obj);
			}
		}
	}
	if (amount > 0) // - 
	{
		sprintf(buf1, ": %s \r\n", diag_liquid_timer(obj)); //  
		out += buf1;
	}
	SendMsgToChar(out, ch);
}

char *daig_filling_drink(const ObjData *obj, const CharData *ch) {
	char tmp[256];
	if (GET_OBJ_VAL(obj, 1) <= 0) {
		sprintf(buf1, "");
		return buf1;
	}
	else {
		if (GET_OBJ_VAL(obj, 0) <= 0 || GET_OBJ_VAL(obj, 1) > GET_OBJ_VAL(obj, 0)) {
			sprintf(buf1, "%s ?!", GET_OBJ_SUF_6(obj));    // BUG
			return buf1;
		}
		else {
			const char *msg = AFF_FLAGGED(ch, EAffect::kDetectPoison) && obj->get_val(3) == 1 ? " **" : "";
			int amt = (GET_OBJ_VAL(obj, 1) * 5) / GET_OBJ_VAL(obj, 0);
			sprinttype(GET_OBJ_VAL(obj, 2), color_liquid, tmp);
			snprintf(buf1, kMaxStringLength,
					 "%s %s%s%s ", GET_OBJ_SUF_6(obj), fullness[amt], tmp, msg);
			return buf1;
		}
	}
}

const char *diag_liquid_timer(const ObjData *obj) {
	int tm;
	if (GET_OBJ_VAL(obj, 3) == 1)
		return "!";
	if (GET_OBJ_VAL(obj, 3) == 0)
		return ".";
	tm = (GET_OBJ_VAL(obj, 3));
	if (tm < 1440) // 
		return " !";
	else if (tm < 10080) //
		return ".";
	else if (tm < 20160) // 2 
		return " .";
	else if (tm < 30240) // 3 
		return ".";
	return ".";
}

} // namespace drinkcon

// vim: ts=4 sw=4 tw=0 noet syntax=cpp :
