/**
\file damage.cpp - a part of the Bylins engine.
\authors Created by Sventovit.
\date 05.11.2025.
\brief Brief description.
\detail Detail description.
*/

#include "damage.h"

#include "third_party_libs/fmt/include/fmt/format.h"

#include "gameplay/mechanics/bonus.h"
#include "engine/entities/char_data.h"
#include "utils/utils_time.h"
#include "engine/db/global_objects.h"
// \TODO    fight.h    . , ,     .
#include "gameplay/fight/fight.h"
#include "gameplay/mechanics/equipment.h"
#include "gameplay/clans/house_exp.h"
#include "gameplay/statistics/dps.h"
#include "engine/ui/color.h"
#include "gameplay/core/game_limits.h"
#include "engine/core/utils_char_obj.inl"
#include "gameplay/affects/affect_handler.h"
#include "gameplay/magic/magic_utils.h"
#include "gameplay/ai/spec_procs.h"
#include "gameplay/mechanics/groups.h"
#include "gameplay/mechanics/tutelar.h"
#include "gameplay/mechanics/sight.h"

void TryRemoveExtrahits(CharData *ch, CharData *victim);

// Estern -  ,         
int max_exp_gain_pc(CharData *ch);
//int max_exp_loss_pc(CharData *ch);

// message for doing damage with a weapon
void Damage::SendDmgMsg(CharData *ch, CharData *victim) const {
	int dam_msgnum;
	int w_type = msg_num;

	static const struct dam_weapon_type {
	  const char *to_room;
	  const char *to_char;
	  const char *to_victim;
	} dam_weapons[] =
		{

			// use #w for singular (i.e. "slash") and #W for plural (i.e. "slashes")

			{
				"$n $u #W $N3,  $u.",    // 0: 0      0 //
				"  #W $N3,  .",
				"$n $u #W ,  $u."
			}, {
				"$n  #w$g $N3.",    //  1..5 1 //
				"  #w $N3.",
				"$n  #w$g ."
			}, {
				"$n  #w$g $N3.",    //  6..11  2 //
				"  #w $N3.",
				"$n  #w$g ."
			}, {
				"$n #w$g $N3.",    //  12..18   3 //
				" #w $N3.",
				"$n #w$g ."
			}, {
				"$n #w$g $N3.",    // 19..26  4 //
				" #w $N3.",
				"$n #w$g ."
			}, {
				"$n  #w$g $N3.",    // 27..35  5 //
				"  #w $N3.",
				"$n  #w$g ."
			}, {
				"$n   #w$g $N3.",    //  36..45 6  //
				"   #w $N3.",
				"$n   #w$g ."
			}, {
				"$n   #w$g $N3.",    //  46..55  7 //
				"   #w $N3.",
				"$n   #w$g ."
			}, {
				"$n  #w$g $N3.",    //  56..96   8 //
				"  #w $N3.",
				"$n  #w$g ."
			}, {
				"$n   #w$g $N3.",    //    97..136  9  //
				"   #w $N3.",
				"$n   #w$g ."
			}, {
				"$n   #w$g $N3.",    //   137..176  10 //
				"   #w $N3.",
				"$n   #w$g ."
			}, {
				"$n   #w$g $N3.",    //    177..216  11 //
				"   #w $N3.",
				"$n   #w$g ."
			}, {
				"$n  #w$g $N3.",    //    217..256  13 //
				"  #w $N3.",
				"$n  #w$g ."
			}, {
				"$n  #w$g $N3.",    //    257..296  13 //
				"  #w $N3.",
				"$n  #w$g ."
			}, {
				"$n  #w$g $N3.",    //    297..400  15 //
				"  #w $N3.",
				"$n  #w$g ."
			}, {
				"$n  #w$g $N3.",    //    297..400  15 //
				"  #w $N3.",
				"$n  #w$g ."
			}, {
				"$n  #w$g $N3.",    // 400+  16 //
				"  #w $N3.",
				"$n  #w$g ."
			}
		};

	if (w_type >= kTypeHit && w_type < kTypeMagic)
		w_type -= kTypeHit;    // Change to base of table with text //
	else
		w_type = kTypeHit;

	if (dam == 0)
		dam_msgnum = 0;
	else if (dam <= 5)
		dam_msgnum = 1;
	else if (dam <= 11)
		dam_msgnum = 2;
	else if (dam <= 18)
		dam_msgnum = 3;
	else if (dam <= 26)
		dam_msgnum = 4;
	else if (dam <= 35)
		dam_msgnum = 5;
	else if (dam <= 45)
		dam_msgnum = 6;
	else if (dam <= 56)
		dam_msgnum = 7;
	else if (dam <= 96)
		dam_msgnum = 8;
	else if (dam <= 136)
		dam_msgnum = 9;
	else if (dam <= 176)
		dam_msgnum = 10;
	else if (dam <= 216)
		dam_msgnum = 11;
	else if (dam <= 256)
		dam_msgnum = 12;
	else if (dam <= 296)
		dam_msgnum = 13;
	else if (dam <= 400)
		dam_msgnum = 14;
	else if (dam <= 800)
		dam_msgnum = 15;
	else
		dam_msgnum = 16;
	// damage message to onlookers
	char *buf_ptr = replace_string(dam_weapons[dam_msgnum].to_room,
								   attack_hit_text[w_type].singular, attack_hit_text[w_type].plural);
	if (brief_shields_.empty()) {
		act(buf_ptr, false, ch, nullptr, victim, kToNotVict | kToArenaListen);
	} else {
		char buf_[kMaxInputLength];
		snprintf(buf_, sizeof(buf_), "%s%s", buf_ptr, brief_shields_.c_str());
		act(buf_, false, ch, nullptr, victim, kToNotVict | kToArenaListen | kToBriefShields);
		act(buf_ptr, false, ch, nullptr, victim, kToNotVict | kToArenaListen | kToNoBriefShields);
	}

	// damage message to damager
	SendMsgToChar(ch, "%s", dam ? "&Y&q" : "&y&q");
	if (!brief_shields_.empty() && ch->IsFlagged(EPrf::kBriefShields)) {
		char buf_[kMaxInputLength];
		snprintf(buf_, sizeof(buf_), "%s%s",
				 replace_string(dam_weapons[dam_msgnum].to_char,
								attack_hit_text[w_type].singular, attack_hit_text[w_type].plural),
				 brief_shields_.c_str());
		act(buf_, false, ch, nullptr, victim, kToChar);
	} else {
		buf_ptr = replace_string(dam_weapons[dam_msgnum].to_char,
								 attack_hit_text[w_type].singular, attack_hit_text[w_type].plural);
		act(buf_ptr, false, ch, nullptr, victim, kToChar);
	}
	SendMsgToChar("&Q&n", ch);

	// damage message to damagee
	SendMsgToChar("&R&q", victim);
	if (!brief_shields_.empty() && victim->IsFlagged(EPrf::kBriefShields)) {
		char buf_[kMaxInputLength];
		snprintf(buf_, sizeof(buf_), "%s%s",
				 replace_string(dam_weapons[dam_msgnum].to_victim,
								attack_hit_text[w_type].singular, attack_hit_text[w_type].plural),
				 brief_shields_.c_str());
		act(buf_, false, ch, nullptr, victim, kToVict | kToSleep);
	} else {
		buf_ptr = replace_string(dam_weapons[dam_msgnum].to_victim,
								 attack_hit_text[w_type].singular, attack_hit_text[w_type].plural);
		act(buf_ptr, false, ch, nullptr, victim, kToVict | kToSleep);
	}
	SendMsgToChar("&Q&n", victim);
}

bool Damage::CalcMagisShieldsDmgAbsoption(CharData *ch, CharData *victim) {
	if (dam <= 0) {
		return false;
	}

	//      
	if (AFF_FLAGGED(victim, EAffect::kMagicGlass)
		&& dmg_type == fight::kMagicDmg) {
		int pct = 6;
		if (victim->IsNpc() && !IS_CHARMICE(victim)) {
			pct += 2;
			if (victim->get_role(static_cast<unsigned>(EMobClass::kBoss))) {
				pct += 2;
			}
		}
		//  
		const int mg_damage = dam * pct / 100;
		if (mg_damage > 0
			&& victim->GetEnemy()
			&& victim->GetPosition() > EPosition::kStun
			&& victim->in_room != kNowhere) {
			flags.set(fight::kDrawBriefMagMirror);
			Damage dmg(SpellDmg(ESpell::kMagicGlass), mg_damage, fight::kUndefDmg);
			dmg.flags.set(fight::kNoFleeDmg);
			dmg.flags.set(fight::kMagicReflect);
			dmg.Process(victim, ch);
		}
	}

	//  ,  Damage::post_init_shields()
	if (flags[fight::kVictimFireShield]
		&& !flags[fight::kCritHit]) {
		if (dmg_type == fight::kPhysDmg
			&& !flags[fight::kIgnoreFireShield]) {
			int pct = 15;
			if (victim->IsNpc() && !IS_CHARMICE(victim)) {
				pct += 5;
				if (victim->get_role(static_cast<unsigned>(EMobClass::kBoss))) {
					pct += 5;
				}
			}
			fs_damage = dam * pct / 100;
		} else {
			act("   $N1   .",
				false, ch, nullptr, victim, kToChar | kToNoBriefShields);
			act("      .",
				false, ch, nullptr, victim, kToVict | kToNoBriefShields);
			act("   $N1   $n1.",
				true, ch, nullptr, victim, kToNotVict | kToArenaListen | kToNoBriefShields);
		}
		flags.set(fight::kDrawBriefFireShield);
		dam -= (dam * number(30, 50) / 100);
	}

	//    (   )    - 95%   
	//         
	if (dam
		&& flags[fight::kCritHit] && flags[fight::kVictimIceShield]
		&& !dam_critic
		&& spell_id != ESpell::kPoison
		&& number(0, 100) < 94) {
		act("         $N1.",
			false, ch, nullptr, victim, kToChar | kToNoBriefShields);
		act("       .",
			false, ch, nullptr, victim, kToVict | kToNoBriefShields);
		act("   $N1     $n1.",
			true, ch, nullptr, victim, kToNotVict | kToArenaListen | kToNoBriefShields);

		flags.reset(fight::kCritHit);
		if (dam > 0) dam -= (dam * number(30, 50) / 100);
	}
		//    
	else if (dam > 0
		&& flags[fight::kVictimIceShield]
		&& !flags[fight::kCritHit]) {
		flags.set(fight::kDrawBriefIceShield);
		act("   $N1   .",
			false, ch, nullptr, victim, kToChar | kToNoBriefShields);
		act("      .",
			false, ch, nullptr, victim, kToVict | kToNoBriefShields);
		act("   $N1   $n1.",
			true, ch, nullptr, victim, kToNotVict | kToArenaListen | kToNoBriefShields);
		dam -= (dam * number(30, 50) / 100);
	}

	if (dam > 0
		&& flags[fight::kVictimAirShield]
		&& !flags[fight::kCritHit]) {
		flags.set(fight::kDrawBriefAirShield);
		act("   $N1   .",
			false, ch, nullptr, victim, kToChar | kToNoBriefShields);
		act("    $n1.",
			false, ch, nullptr, victim, kToVict | kToNoBriefShields);
		act("   $N1   $n1.",
			true, ch, nullptr, victim, kToNotVict | kToArenaListen | kToNoBriefShields);
		dam -= (dam * number(30, 50) / 100);
	}

	return false;
}

void Damage::CalcArmorDmgAbsorption(CharData *victim) {
	//    
	if (dam > 0 && dmg_type == fight::kPhysDmg) {
		DamageEquipment(victim, kNowhere, dam, 50);
		if (!flags[fight::kCritHit] && !flags[fight::kIgnoreArmor]) {
			// 50  = 50%  
			int max_armour = 50;
			if (CanUseFeat(victim, EFeat::kImpregnable) && victim->IsFlagged(EPrf::kAwake)) {
				//    -  75 
				max_armour = 75;
			}
			int tmp_dam = dam * std::max(0, std::min(max_armour, GET_ARMOUR(victim))) / 100;
			//     
			if (tmp_dam >= 2 && flags[fight::kHalfIgnoreArmor]) {
				tmp_dam /= 2;
			}
			dam -= tmp_dam;
			//    ,       .
		}
	}
}

/**
 *    .
 * \return true -  
 */
bool Damage::CalcDmgAbsorption(CharData *ch, CharData *victim) {
	if (dmg_type == fight::kPhysDmg
		&& skill_id < ESkill::kFirst
		&& spell_id < ESpell::kUndefined
		&& dam > 0
		&& GET_ABSORBE(victim) > 0) {
		//  :    15%,  10%
		int chance = 10 + GetRealRemort(victim) / 3;
		if (CanUseFeat(victim, EFeat::kImpregnable)
			&& victim->IsFlagged(EPrf::kAwake)) {
			chance += 5;
		}
		//   -    
		if (number(1, 100) <= chance) {
			dam -= GET_ABSORBE(victim) / 2;
			if (dam <= 0) {
				act("     $n1.",
					false, ch, nullptr, victim, kToVict);
				act(" $N1    .",
					false, ch, nullptr, victim, kToChar);
				act(" $N1    $n1.",
					true, ch, nullptr, victim, kToNotVict | kToArenaListen);
				return true;
			}
		}
	}
	if (dmg_type == fight::kMagicDmg
		&& dam > 0
		&& GET_ABSORBE(victim) > 0
		&& !flags[fight::kIgnoreAbsorbe]) {
//   -  1%   2 ,  25% (  mag_damage)
		int absorb = std::min(GET_ABSORBE(victim) / 2, 25);
		dam -= dam * absorb / 100;
	}
	return false;
}

void Damage::SendCritHitMsg(CharData *ch, CharData *victim) {
	//        ,
	//         ()
	if (!flags[fight::kVictimIceShield]) {
		sprintf(buf, "&G&q     %s.&Q&n\r\n",
				PERS(victim, ch, 3));
	} else {
		sprintf(buf, "&B&q        %s.&Q&n\r\n",
				PERS(victim, ch, 1));
	}

	SendMsgToChar(buf, ch);

	if (!flags[fight::kVictimIceShield]) {
		sprintf(buf, "&r&q  %s   .&Q&n\r\n",
				PERS(ch, victim, 1));
	} else {
		sprintf(buf, "&r&q  %s      .&Q&n\r\n",
				PERS(ch, victim, 1));
	}

	SendMsgToChar(buf, victim);
	//    ,     
	//act("  $N1  $n3 .", true, victim, nullptr, ch, TO_NOTVICT);
}

void Damage::ProcessBlink(CharData *ch, CharData *victim) {
	if (flags[fight::kIgnoreBlink] || flags[fight::kCritLuck])
		return;
	ubyte blink = 0;
	//       
	if (dmg_type == fight::kMagicDmg) {
		if (AFF_FLAGGED(victim, EAffect::kCloudly) || victim->add_abils.percent_spell_blink_mag > 0) {
			if (victim->IsNpc()) {
				blink = GetRealLevel(victim) + GetRealRemort(victim);
			} else if(victim->add_abils.percent_spell_blink_mag > 0) {
				blink = victim->add_abils.percent_spell_blink_mag;
			} else {
				blink = 10;
			}
		}
	} else if(dmg_type == fight::kPhysDmg) {
		if (AFF_FLAGGED(victim, EAffect::kBlink) || victim->add_abils.percent_spell_blink_phys > 0) {
			if (victim->IsNpc()) {
				blink = GetRealLevel(victim) + GetRealRemort(victim);
			} else if (victim->add_abils.percent_spell_blink_phys > 0) {
				blink = victim->add_abils.percent_spell_blink_phys;
			} else {
				blink = 10;
			}
		}
	}
	if(blink < 1)
		return;
//	ch->send_to_TC(false, true, false, "   == %d .\r\n", blink);
//	victim->send_to_TC(false, true, false, "   == %d .\r\n", blink);
	int bottom = 1;
	if (ch->calc_morale() > number(1, 100)) // 
		bottom = 10;
	if (number(bottom, blink) >= number(1, 100)) {
		sprintf(buf, "%s       .%s\r\n",
				kColorBoldBlk, kColorNrm);
		SendMsgToChar(buf, victim);
		act("$n $q    .", true, victim, nullptr, ch, kToVict);
		act("$n $q    $N1.", true, victim, nullptr, ch, kToNotVict);
		dam = 0;
		fs_damage = 0;
		return;
	}
}

void Damage::ProcessDeath(CharData *ch, CharData *victim) const {
	CharData *killer = nullptr;

	if (victim->IsNpc() || victim->desc) {
		if (victim == ch && victim->in_room != kNowhere) {
			if (spell_id == ESpell::kPoison) {
				for (const auto poisoner : world[victim->in_room]->people) {
					if (poisoner != victim
						&& poisoner->get_uid() == victim->poisoner) {
						killer = poisoner;
					}
				}
			} else if (msg_num == kTypeSuffering) {
				for (const auto attacker : world[victim->in_room]->people) {
					if (attacker->GetEnemy() == victim) {
						killer = attacker;
					}
				}
			}
		}

		if (ch != victim) {
			killer = ch;
		}
	}
	if (killer) {
		if (AFF_FLAGGED(killer, EAffect::kGroup)) {
			// ..   AFF_GROUP -  PC
			group_gain(killer, victim);
		} else if ((AFF_FLAGGED(killer, EAffect::kCharmed)
			|| killer->IsFlagged(EMobFlag::kTutelar)
			|| killer->IsFlagged(EMobFlag::kMentalShadow))
			&& killer->has_master())
			// killer -  NPC  
		{
			//     ,      , 
			// -     ,    ,
			//      .
			if (AFF_FLAGGED(killer->get_master(), EAffect::kGroup)
				&& killer->in_room == killer->get_master()->in_room) {
				//  - PC   =>  
				group_gain(killer->get_master(), victim);
			} else if (killer->in_room == killer->get_master()->in_room) {
				perform_group_gain(killer->get_master(), victim, 1, 100);
			}
			// else
			//      ,   - 
			//     group::perform_group_gain( killer, victim, 1, 100 );
		} else {
			//  NPC  PC   
			perform_group_gain(killer, victim, 1, 100);
		}
	}

	//         ( )
	//      
	//        
	if (!victim->IsNpc() && !(killer && killer->IsFlagged(EPrf::kExecutor))) {
		UpdatePkLogs(ch, victim);

		for (const auto &ch_vict : world[ch->in_room]->people) {
			//       
			if (IS_IMMORTAL(ch_vict))
				continue;
			if (!HERE(ch_vict))
				continue;
			if (!ch_vict->IsNpc())
				continue;
			if (ch_vict->IsFlagged(EMobFlag::kMemory)) {
				mob_ai::mobForget(ch_vict, victim);
			}
		}

	}

	if (killer) {
		ch = killer;
	}
	die(victim, ch);
}

/**
 *       .
 *     3 ,    1    .
 */
void Damage::SetPostInitShieldFlags(CharData *victim) {
	if (victim->IsNpc() && !IS_CHARMICE(victim)) {
		if (AFF_FLAGGED(victim, EAffect::kFireShield)) {
			flags.set(fight::kVictimFireShield);
		}

		if (AFF_FLAGGED(victim, EAffect::kIceShield)) {
			flags.set(fight::kVictimIceShield);
		}

		if (AFF_FLAGGED(victim, EAffect::kAirShield)) {
			flags.set(fight::kVictimAirShield);
		}
	} else {
		enum { FIRESHIELD, ICESHIELD, AIRSHIELD };
		std::vector<int> shields;

		if (AFF_FLAGGED(victim, EAffect::kFireShield)) {
			shields.push_back(FIRESHIELD);
		}

		if (AFF_FLAGGED(victim, EAffect::kAirShield)) {
			shields.push_back(AIRSHIELD);
		}

		if (AFF_FLAGGED(victim, EAffect::kIceShield)) {
			shields.push_back(ICESHIELD);
		}

		if (shields.empty()) {
			return;
		}

		int shield_num = number(0, static_cast<int>(shields.size() - 1));

		if (shields[shield_num] == FIRESHIELD) {
			flags.set(fight::kVictimFireShield);
		} else if (shields[shield_num] == AIRSHIELD) {
			flags.set(fight::kVictimAirShield);
		} else if (shields[shield_num] == ICESHIELD) {
			flags.set(fight::kVictimIceShield);
		}
	}
}

void Damage::PerformPostInit(CharData *ch, CharData *victim) {
	if (msg_num == -1) {
		// ABYRVALG         
		if (MUD::Skills().IsValid(skill_id)) {
			msg_num = to_underlying(skill_id) + kTypeHit;
		} else if (spell_id > ESpell::kUndefined) {
			msg_num = to_underlying(spell_id);
		} else if (hit_type >= 0) {
			msg_num = hit_type + kTypeHit;
		} else {
			msg_num = kTypeHit;
		}
	}

	if (ch_start_pos == EPosition::kUndefined) {
		ch_start_pos = ch->GetPosition();
	}

	if (victim_start_pos == EPosition::kUndefined) {
		victim_start_pos = victim->GetPosition();
	}

	SetPostInitShieldFlags(victim);
}

//  , , ,   .   
//   
int Damage::Process(CharData *ch, CharData *victim) {
	PerformPostInit(ch, victim);
	if (!check_valid_chars(ch, victim, __FILE__, __LINE__)) {
		return 0;
	}
	if (victim->in_room == kNowhere || ch->in_room == kNowhere || ch->in_room != victim->in_room) {
		log("SYSERR: Attempt to damage '%s' in room kNowhere by '%s'.",
			GET_NAME(victim), GET_NAME(ch));
		return 0;
	}
	if (victim->GetPosition() <= EPosition::kDead) {
		log("SYSERR: Attempt to damage corpse '%s' in room #%d by '%s'.",
			GET_NAME(victim), GET_ROOM_VNUM(victim->in_room), GET_NAME(ch));
		die(victim, nullptr);
		return 0;
	}
	if ((ch->IsNpc() && ch->IsFlagged(EMobFlag::kNoFight))
		|| (victim->IsNpc() && victim->IsFlagged(EMobFlag::kNoFight))) {
		return 0;
	}
	if (dam > 0) {
		if (IS_GOD(victim)) {
			dam = 0;
		} else if (IS_IMMORTAL(victim) || GET_GOD_FLAG(victim, EGf::kGodsLike)) {
			dam /= 4;
		} else if (GET_GOD_FLAG(victim, EGf::kGodscurse)) {
			dam *= 2;
		}
	}

	mob_ai::update_mob_memory(ch, victim);
	Appear(ch);
	Appear(victim);

	// If you attack a pet, it hates your guts
	if (!group::same_group(ch, victim)) {
		check_agro_follower(ch, victim);
	}
	if (victim != ch) {
		if (ch->GetPosition() > EPosition::kStun && (ch->GetEnemy() == nullptr)) {
			if (!pk_agro_action(ch, victim)) {
				return 0;
			}
			SetFighting(ch, victim);
			npc_groupbattle(ch);
		}
		// Start the victim fighting the attacker
		if (victim->GetPosition() > EPosition::kDead && (victim->GetEnemy() == nullptr)) {
			SetFighting(victim, ch);
			npc_groupbattle(victim);
		}

		//     
		if (ch->IsOnHorse() && ch->get_horse() == victim) {
			victim->DropFromHorse();
		} else if (victim->IsOnHorse() && victim->get_horse() == ch) {
			ch->DropFromHorse();
		}
	}

	if (dam < 0 || ch->in_room == kNowhere || victim->in_room == kNowhere || ch->in_room != victim->in_room) {
		return 0;
	}

	if (ch->GetPosition() <= EPosition::kIncap) {
		return 0;
	}

	if (dam >= 2) {
		if (AFF_FLAGGED(victim, EAffect::kPrismaticAura) && !flags[fight::kIgnorePrism]) {
			if (dmg_type == fight::kPhysDmg) {
				dam *= 2;
			} else if (dmg_type == fight::kMagicDmg) {
				dam /= 2;
			}
		}
		if (AFF_FLAGGED(victim, EAffect::kSanctuary) && !flags[fight::kIgnoreSanct]) {
			if (dmg_type == fight::kPhysDmg) {
				dam /= 2;
			} else if (dmg_type == fight::kMagicDmg) {
				dam *= 2;
			}
		}

		if (victim->IsNpc() && Bonus::is_bonus_active(Bonus::EBonusType::BONUS_DAMAGE)) {
			dam *= Bonus::get_mult_bonus();
		}
	}
	//    
	if (dmg_type == fight::kMagicDmg) {
		if (spell_id > ESpell::kUndefined) {
			dam = ApplyResist(victim, GetResistType(spell_id), dam);
		} else {
			dam = ApplyResist(victim, GetResisTypeWithElement(element), dam);
		};
	};

	//     
	// Include a damage multiplier if victim isn't ready to fight:
	// Position sitting  1.5 x normal
	// Position resting  2.0 x normal
	// Position sleeping 2.5 x normal
	// Position stunned  3.0 x normal
	// Position incap    3.5 x normal
	// Position mortally 4.0 x normal
	// Note, this is a hack because it depends on the particular
	// values of the POSITION_XXX constants.

	//      
	if (ch_start_pos < EPosition::kFight && dmg_type == fight::kPhysDmg) {
		dam -= dam * (EPosition::kFight - ch_start_pos) / 4;
	}

	//    :
	//     
	//  -   ( mage_damage        )
	if (victim_start_pos < EPosition::kFight
		&& !flags[fight::kVictimAirShield]
		&& !(dmg_type == fight::kMagicDmg
			&& ch->IsNpc())) {
		dam += dam * (EPosition::kFight - victim_start_pos) / 4;
	}

	//  

	if (AFF_FLAGGED(victim, EAffect::kHold) && dmg_type == fight::kPhysDmg) {
		if (ch->IsNpc() && !IS_CHARMICE(ch)) {
			dam = dam * 15 / 10;
		} else {
			dam = dam * 125 / 100;
		}
	}

	if (!victim->IsNpc() && IS_CHARMICE(ch)) {
		dam = dam * 8 / 10;
	}

	if (GET_PR(victim) && dmg_type == fight::kPhysDmg) {
		int ResultDam = dam - (dam * GET_PR(victim) / 100);
		ch->send_to_TC(false, true, false,
					   "&C  : %d , %d .&n\r\n", dam, ResultDam);
		victim->send_to_TC(false, true, false,
						   "&C  : %d , %d .&n\r\n", dam, ResultDam);
		dam = ResultDam;
	}
	if (!IS_IMMORTAL(ch) && AFF_FLAGGED(victim, EAffect::kGodsShield)) {
		if (skill_id == ESkill::kBash) {
			SendSkillMessages(dam, ch, victim, msg_num);
		}
		if (ch != victim) {
			act("     $N1.", false, victim, nullptr, ch, kToChar);
			act("   $N1    .", false, ch, nullptr, victim, kToChar);
			act("   $N1    $n1.", true, ch, nullptr, victim, kToNotVict | kToArenaListen);
		} else {
			act("    .", false, ch, nullptr, nullptr, kToChar);
			act("   $N1   .", true, ch, nullptr, victim, kToNotVict | kToArenaListen);
		}
		return 0;
	}
	// , , 
	if (victim != ch) {
		bool shield_full_absorb = CalcMagisShieldsDmgAbsoption(ch, victim);
		CalcArmorDmgAbsorption(victim);
		bool armor_full_absorb = CalcDmgAbsorption(ch, victim);
		if (flags[fight::kCritHit] && (GetRealLevel(victim) >= 5 || !ch->IsNpc())
			&& !AFF_FLAGGED(victim, EAffect::kPrismaticAura)
			&& !flags[fight::kVictimIceShield]) {
			int tmpdam = std::min(victim->get_real_max_hit() / 8, dam * 2);
			tmpdam = ApplyResist(victim, EResist::kVitality, dam);
			dam = std::max(dam, tmpdam); //
		}
		//  
		if (shield_full_absorb || armor_full_absorb) {
			return 0;
		}
		if (dam > 0)
			ProcessBlink(ch, victim);
	}

	//  magic_shields_dam  dmg::proccess,    ,   
	if (!(ch && victim) || (ch->purged() || victim->purged())) {
		log("Death from CalcMagisShieldsDmgAbsoption");
		return 0;
	}

	if (victim->IsFlagged(EMobFlag::kProtect)) {
		if (victim != ch) {
			act("$n    .", false, victim, nullptr, nullptr, kToRoom);
		}
		return 0;
	}
	//   ! !,     -  
	DamageActorParameters params(ch, victim, dam);
	handle_affects(params);
	dam = params.damage;
	DamageVictimParameters params1(ch, victim, dam);
	handle_affects(params1);
	dam = params1.damage;

	//   / 
	if (flags[fight::kMagicReflect]) {
		//     40%    
		dam = std::min(dam, victim->get_max_hit() * 4 / 10);
		//    
		dam = std::min(dam, victim->get_hit() - 1);
	}

	dam = std::clamp(dam, 0, kMaxHits);
	if (dam >= 0) {
		if (dmg_type == fight::kPhysDmg) {
			if (!damage_mtrigger(ch, victim, dam, MUD::Skill(skill_id).GetName(), 1, wielded))
				return 0;
		} else if (dmg_type == fight::kMagicDmg) {
			if (!damage_mtrigger(ch, victim, dam, MUD::Spell(spell_id).GetCName(), 0, wielded))
				return 0;
		} else if (dmg_type == fight::kPoisonDmg) {
			if (!damage_mtrigger(ch, victim, dam, MUD::Spell(spell_id).GetCName(), 2, wielded))
				return 0;
		}
	}
	if (!InTestZone(ch)) {
		gain_battle_exp(ch, victim, dam);
	}

	// real_dam       .
	int real_dam = dam;
	int over_dam = 0;

	if (dam > victim->get_hit() + 11) {
		real_dam = victim->get_hit() + 11;
		over_dam = dam - real_dam;
	}
	//   
	victim->set_hit(victim->get_hit() - dam);
	victim->send_to_TC(false, true, true, "&M  = %d&n\r\n", dam);
	ch->send_to_TC(false, true, true, "&M  = %d&n\r\n", dam);
	if (dmg_type == fight::kPhysDmg && GET_GOD_FLAG(ch, EGf::kSkillTester) && skill_id != ESkill::kUndefined) {
		log("SKILLTEST:;%s;skill;%s;damage;%d;Luck;%s", GET_NAME(ch), MUD::Skill(skill_id).GetName(), dam, flags[fight::kCritLuck] ? "yes" : "no");
	}
	//    
	if (AFF_FLAGGED(ch, EAffect::kVampirism)) {
		ch->set_hit(std::clamp(ch->get_hit() + std::max(1, dam / 10),
							   ch->get_hit(), ch->get_real_max_hit() + ch->get_real_max_hit() * GetRealLevel(ch) / 10));
		//    ,     5%    
		if (ch->has_master()) {
			if (CanUseFeat(ch->get_master(), EFeat::kSoulLink)) {
				ch->get_master()->set_hit(std::max(ch->get_master()->get_hit(),
												   std::min(ch->get_master()->get_hit() + std::max(1, dam / 20 ),
															ch->get_master()->get_real_max_hit() +
																ch->get_master()->get_real_max_hit() *
																	GetRealLevel(ch->get_master()) / 10)));
			}
		}
	}
	//       
	DpsSystem::UpdateDpsStatistics(ch, real_dam, over_dam);
	//     
	if (victim->IsNpc()) {
		victim->add_attacker(ch, ATTACKER_DAMAGE, real_dam);
	}
	//     
	CheckTutelarSelfSacrfice(ch, victim);

	//      
	update_pos(victim);
	//      ,   ,    
	if (!(ch && victim) || (ch->purged() || victim->purged())) {
		log("Error in fight_hit, function process()\r\n");
		return 0;
	}
	//      
	if (CanUseFeat(victim, EFeat::kHarvestOfLife)) {
		if (victim->GetPosition() == EPosition::kDead) {
			int souls = victim->get_souls();
			if (souls >= 10) {
				victim->set_hit(0 + souls * 10);
				update_pos(victim);
				SendMsgToChar("&G    !&n\r\n", victim);
				victim->set_souls(0);
			}
		}
	}
	//    //
	if (spell_id != ESpell::kPoison && dam > 0 && !flags[fight::kMagicReflect]) {
		TryRemoveExtrahits(ch, victim);
	}

	if (dam && flags[fight::kCritHit] && !dam_critic && spell_id != ESpell::kPoison) {
		SendCritHitMsg(ch, victim);
	}

	//  Damage::brief_shields_
	if (flags.test(fight::kDrawBriefFireShield)
		|| flags.test(fight::kDrawBriefAirShield)
		|| flags.test(fight::kDrawBriefIceShield)
		|| flags.test(fight::kDrawBriefMagMirror)) {
		char buf_[kMaxInputLength];
		snprintf(buf_, sizeof(buf_), "&n (%s%s%s%s)",
				 flags.test(fight::kDrawBriefFireShield) ? "&R*&n" : "",
				 flags.test(fight::kDrawBriefAirShield) ? "&C*&n" : "",
				 flags.test(fight::kDrawBriefIceShield) ? "&W*&n" : "",
				 flags.test(fight::kDrawBriefMagMirror) ? "&M*&n" : "");
		brief_shields_ = buf_;
	}
	//    //
	if (MUD::Skills().IsValid(skill_id) || spell_id > ESpell::kUndefined || hit_type < 0) {
		// , ,  
		SendSkillMessages(dam, ch, victim, msg_num, brief_shields_);
	} else {
		//   /
		if (victim->GetPosition() == EPosition::kDead || dam == 0) {
			if (!SendSkillMessages(dam, ch, victim, msg_num, brief_shields_)) {
				SendDmgMsg(ch, victim);
			}
		} else {
			SendDmgMsg(ch, victim);
		}
	}
	/// Use SendMsgToChar -- act() doesn't send message if you are DEAD.
	char_dam_message(dam, ch, victim, flags[fight::kNoFleeDmg]);


	// ,     .     .
	// ,   .
	// ,    FIRESHIELD,
	//       
	if (ch->in_room != victim->in_room) {
		return dam;
	}

	// Stop someone from fighting if they're stunned or worse
	/*if ((victim->GetPosition() <= EPosition::kStun)
		&& (victim->GetEnemy() != NULL))
	{
		stop_fighting(victim, victim->GetPosition() <= EPosition::kDead);
	} */
	utils::CSteppedProfiler round_profiler(fmt::format("process death"), 0.01);
	round_profiler.next_step("Start");

	//   //
	if (victim->GetPosition() == EPosition::kDead) {
		ProcessDeath(ch, victim);
		return -1;
	}
	//    
	if (fs_damage > 0
		&& victim->GetEnemy()
		&& victim->GetPosition() > EPosition::kStun
		&& victim->in_room != kNowhere) {
		Damage dmg(SpellDmg(ESpell::kFireShield), fs_damage, fight::kUndefDmg);
		dmg.flags.set(fight::kNoFleeDmg);
		dmg.flags.set(fight::kMagicReflect);
		dmg.Process(victim, ch);
	}
	return dam;
}

// *     1.5    1% ,      .
void TryRemoveExtrahits(CharData *ch, CharData *victim) {
	if (((!ch->IsNpc() && ch != victim)
		|| (ch->has_master()
			&& !ch->get_master()->IsNpc()
			&& ch->get_master() != victim))
		&& !victim->IsNpc()
		&& victim->GetPosition() != EPosition::kDead
		&& victim->get_hit() > victim->get_real_max_hit() * 1.5
		&& number(1, 100) == 5)//   5,   1     - 
	{
		victim->set_hit(victim->get_real_max_hit());
		SendMsgToChar(victim, "%s'%s %s  ' -      .%s\r\n",
					  kColorWht, GET_CH_POLY_1(victim), GET_CH_EXSUF_1(victim), kColorNrm);
		act("   ,  $N3 .", false, ch, nullptr, victim, kToChar);
		act("$n $g  ,  $N3 .", false, ch, nullptr, victim, kToNotVict | kToArenaListen);
	}
}

// vim: ts=4 sw=4 tw=0 noet syntax=cpp :
