// $RCSfile$     $Date$     $Revision$
// Copyright (c) 2007 Krodo
// Part of Bylins http://www.mud.ru

#include "privilege.h"

#include "utils/logger.h"
#include "engine/entities/char_data.h"
#include "gameplay/communication/boards/boards.h"
#include "engine/db/player_index.h"

#define TEST_BUILD // prool:    ,    : 34   ӣ

/**
*     ,    god.lst.
*         ->        .
*          .
*   privilege.lst:
* #   (  ): default, default_demigod, boards, arena, skills
* # default - ,      (   )
* # default_demigod -     
* # boards -     , ,      
* # arena - ,     ,  set  show (+    )
* # skills - ,    , , ,   (34  )
* # fullzedit -  /    (zed lock/unlock) (34  )
* # title - /  
* #     
* # :   (0  ,     ) /
* <groups>
* default = wizhelp wiznet register   title holylight uptime date set (title name) rules show nohassle ; show (punishment stats player)
* default_demigod = wizhelp wiznet  rules
* arena = purge
* olc = oedit zedit redit olc trigedit
* goto = goto  poofin poofout
* </groups>
* <gods>
*  595336650 groups (olc) hell mute dumb ban delete set (bank)
*  803863739 groups (arena goto olc boards) hell mute dumb ban delete set (bank)
* </gods>
*   , zone.ru      lua,  xml      ,   ...
*/
namespace privilege {

const int kBoards = 0;
const int kUseSkills = 1;
const int kArenaMaster = 2;
const int kKroder = 3;
const int kFullzedit = 4;
const int kTitle = 5;
// /  ,     olc
const int kMisprint = 6;
// /  
const int kSuggest = 7;
//  
const int FLAGS_NUM = 8;

typedef std::set<std::string> PrivListType;

class GodListNode {
 public:
	std::string name; // 
	PrivListType set; //   set
	PrivListType show; //   show
	PrivListType other; //  
	PrivListType arena; //    
	std::bitset<FLAGS_NUM> flags; // 
	void clear() {
		name.clear();
		set.clear();
		show.clear();
		other.clear();
		arena.clear();
		flags.reset();
	}
};

const char *PRIVILEGE_FILE = LIB_MISC"privilege.lst";
typedef std::map<long, GodListNode> GodListType;
GodListType god_list; //     
std::map<std::string, std::string> group_list; //  ,   (    )
GodListNode tmp_god; //  
void parse_command_line(const std::string &command, int other_flags = 0); // 

/**
*       (      ),
*        .
* \param command -  
*/
void parse_flags(const std::string &command) {
	if (command == "boards")
		tmp_god.flags.set(kBoards);
	else if (command == "skills")
		tmp_god.flags.set(kUseSkills);
	else if (command == "arena")
		tmp_god.flags.set(kArenaMaster);
	else if (command == "kroder")
		tmp_god.flags.set(kKroder);
	else if (command == "fullzedit")
		tmp_god.flags.set(kFullzedit);
	else if (command == "title")
		tmp_god.flags.set(kTitle);
	else if (command == "olc")
		tmp_god.flags.set(kMisprint);
	else if (command == "suggest")
		tmp_god.flags.set(kSuggest);
}

/**
*     (, set, show),   arena     
* \param command -  , fill_mode -    , other_flags -        
*/
void insert_command(const std::string &command, int fill_mode, int other_flags) {
	if (other_flags == 1) {
		//       ,       set  show
		if (!fill_mode)
			tmp_god.arena.insert(command);
		return;
	}

	switch (fill_mode) {
		case 0: tmp_god.other.insert(command);
			break;
		case 1: tmp_god.set.insert(command);
			break;
		case 2: tmp_god.show.insert(command);
			break;
		case 3: {
			const auto it = group_list.find(command);
			if (it != group_list.end()) {
				if (command == "arena")
					parse_command_line(it->second, 1);
				else
					parse_command_line(it->second);
			}
			break;
		}
		default: break;
	}
}

// *           default  default_demigod.
void insert_default_command(long uid) {
	std::map<std::string, std::string>::const_iterator it;
	if (GetLevelByUnique(uid) < kLvlImmortal)
		it = group_list.find("default_demigod");
	else
		it = group_list.find("default");
	if (it != group_list.end())
		parse_command_line(it->second);
}


std::vector<std::string> tokenize(const std::string& str) {
	std::vector<std::string> tokens;
	std::istringstream iss(str);
	std::string token;

	while (iss >> token) {
		std::string new_token;

		for (char c : token) {
			if (c == '(' || c == ')') {
				if (!new_token.empty()) {
					tokens.push_back(new_token);
					new_token.clear();
				}
				tokens.push_back(std::string(1, c));
			} else {
				new_token += c;
			}
		}
		if (!new_token.empty()) {
			tokens.push_back(new_token);
		}
	}
	return tokens;
}
/**
*      ,     
* \param other_flags -   0 (     ), 1 -    arena
* \param commands -      
*/
void parse_command_line(const std::string &commands, int other_flags) {
	std::vector<std::string>::iterator tok_iter, tmp_tok_iter;
	std::stringstream ss;
	int fill_mode = 0;
	auto tokens = tokenize(commands);

/*	for (auto it : tokens) {
		ss << it << "|";
	}
	ss << "\r\n";
	mudlog(ss.str(), CMP, kLvlImmortal, SYSLOG, true);
*/
	if (tokens.begin() == tokens.end()) 
		return;
	for (tok_iter = tokens.begin(); tok_iter != tokens.end(); ++tok_iter) {
		if ((*tok_iter) == "(") {
			if ((*tmp_tok_iter) == "set") {
				fill_mode = 1;
				continue;
			} else if ((*tmp_tok_iter) == "show") {
				fill_mode = 2;
				continue;
			} else if ((*tmp_tok_iter) == "groups") {
				fill_mode = 3;
				continue;
			}
		} else if ((*tok_iter) == ")") {
			fill_mode = 0;
			continue;
		}
		parse_flags(*tok_iter);
		insert_command(*tok_iter, fill_mode, other_flags);
		tmp_tok_iter = tok_iter;
	}
}

// *      (reload privilege)     .
void Load() {
	std::ifstream file(PRIVILEGE_FILE);
	if (!file.is_open()) {
		log("Error open file: %s! (%s %s %d)", PRIVILEGE_FILE, __FILE__, __func__, __LINE__);
		return;
	}
	god_list.clear(); //  

	std::string name, commands, temp;
	long uid;

	while (file >> name) {
		if (name == "#") {
			ReadEndString(file);
			continue;
		} else if (name == "<groups>") {

			while (file >> name) {
				if (name == "#") {
					ReadEndString(file);
					continue;
				}
				if (name == "</groups>")
					break;

				file >> temp; // "="
				std::getline(file, commands);
				utils::Trim(commands);
				group_list[name] = commands;
			}
			continue;
		} else if (name == "<gods>") {
			while (file >> name) {
				if (name == "#") {
					ReadEndString(file);
					continue;
				}
				if (name == "</gods>")
					break;
				file >> uid;
				name_convert(name);
				tmp_god.name = name;
				std::getline(file, commands);
				utils::Trim(commands);
				parse_command_line(commands);
				insert_default_command(uid);
				god_list[uid] = tmp_god;
				tmp_god.clear();
			}
		}
	}
	//  
	LoadGodBoards();
	group_list.clear();
}

/**
*          .   CharacterData     .
*      wiz   ,    make test         .
* :      ,   ...
* \param name -  , unique -  
* \return 0 -  , 1 - 
*/

bool IsContainedInGodsList(const std::string &name, long unique) {
#ifdef TEST_BUILD
	return true;
#endif
	auto it = god_list.find(unique);
	if (it != god_list.end())
		if (it->second.name == name)
			return true;
	return false;
}

// *   /  .
void LoadGodBoards() {
	Boards::Static::clear_god_boards();
	for (auto & god : god_list) {
		int level = GetLevelByUnique(god.first);
		if (level < kLvlImmortal) continue;
		Boards::Static::init_god_board(god.first, god.second.name);
	}
}

/**
*      (    31+). 34   .
*    make test        .
* \param mode 0 -  , 1 -  set, 2 -  show
* \return 0 - , 1 - 
*/
bool HasPrivilege(CharData *ch, const std::string &cmd_name, int cmd_number, int mode, bool check_level) {
	if (check_level && !mode && cmd_info[cmd_number].minimum_level < kLvlImmortal
		&& GetRealLevel(ch) >= cmd_info[cmd_number].minimum_level) {
		return true;
	}
	if (ch->IsNpc()) {
		return false;
	}
#ifdef TEST_BUILD
	if (IS_IMMORTAL(ch))
		return true;
#endif
	const auto it = god_list.find(ch->get_uid());
	if (it != god_list.end() && CompareParam(it->second.name, ch->get_name(), true)) {
		if (GetRealLevel(ch) == kLvlImplementator)
			return true;
		switch (mode) {
			case 0:
				if (it->second.other.find(cmd_name) != it->second.other.end())
					return true;
				break;
			case 1:
				if (it->second.set.find(cmd_name) != it->second.set.end())
					return true;
				break;
			case 2:
				if (it->second.show.find(cmd_name) != it->second.show.end())
					return true;
				break;
			default: break;
		}
		//       arena_master
		if (!mode && ROOM_FLAGGED(ch->in_room, ERoomFlag::kArena) && it->second.arena.find(cmd_name) != it->second.arena.end())
			return true;
	}
	return false;
}

/**
*  . 34    skills
*    ,    .
* \param flag -     , - FLAGS_NUM
* \return 0 -  , 1 - 
*/
bool CheckFlag(const CharData *ch, int flag) {
	if (flag >= FLAGS_NUM || flag < 0) return false;
	bool result = false;
	const auto it = god_list.find(ch->get_uid());
	if (it != god_list.end() && CompareParam(it->second.name, GET_NAME(ch), true))
		if (it->second.flags[flag])
			result = true;
	if (flag == kUseSkills && (IS_IMPL(ch)))
		result = true;
	return result;
}

/**
*      .
*  skills  .  arena  ,         .
*    34   .
*/
bool IsSpellPermit(const CharData *ch, ESpell spell_id) {
	if (!IS_IMMORTAL(ch) || IS_IMPL(ch) || CheckFlag(ch, kUseSkills)) {
		return true;
	}
	if (spell_id == ESpell::kPortal || spell_id == ESpell::kSummon || spell_id == ESpell::kWorldOfRecall) {
		if (ROOM_FLAGGED(ch->in_room, ERoomFlag::kArena) && CheckFlag(ch, kArenaMaster)) {
			return true;
		}
	}
	return false;
}

/**
*     .   get_skill.
*  ,   34   .
* \return 0 -    , 1 - 
*/
bool CheckSkills(const CharData *ch) {
	if ((GetRealLevel(ch) > kLvlGod) || !IS_IMMORTAL(ch) || CheckFlag(ch, kUseSkills))
//	if (!IS_IMMORTAL(ch) || IS_IMPL(ch) || check_flag(ch, USE_SKILLS))
		return true;
	return false;
}

} // namespace Privilege

// vim: ts=4 sw=4 tw=0 noet syntax=cpp :
